/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeDataRenderer.h
 *    Renderer plugin base class for rendering data in TapeDataView.
 *
 * @author Tim Langner
 * @date   2012/01/01
 */

#ifndef _MIRA_TAPEDATARENDERER_H_
#define _MIRA_TAPEDATARENDERER_H_

#include <QRect>

#include <factory/Factory.h>
#include <fw/TapeVisitor.h>
#include <utils/EnumToFlags.h>

#include <TapeChannelInfo.h>

class QPainter;
class QMenu;
class QAction;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class TapeEditor;

class TapeDataRenderer : public Object
{
	MIRA_OBJECT(TapeDataRenderer);
public:

	enum PluginFeature
	{
		NONE   = 0x01,
		RENDER = 0x02,
		INFO   = 0x04,
		EDIT   = 0x08,
		CREATE = 0x10,
	};
	MIRA_ENUM_TO_FLAGS_INCLASS(PluginFeature)

	TapeDataRenderer(TapeEditor* editor = NULL) :
		mEditor(editor),
		mDataHeight(1),
		mIndicatorHeight(24),
		mFeatures(INFO | EDIT) {}

	virtual ~TapeDataRenderer() {};

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("IndicatorHeight", mIndicatorHeight, "Height of the data item indicator");
	}

	void setEditor(TapeEditor* editor) { mEditor = editor; }

	PluginFeature getFeatures() const { return mFeatures; }
	bool hasFeature(PluginFeature feature) const { return (mFeatures & feature) > 0; }

	/**
	 * Called when user activates context menu in DataView to fill the menu with
	 * render specific actions. When the user selects one of these actions
	 * executeAction is called.
	 * @param menu The menu to fill
	 * @param info The channel info for the channel the user clicked on
	 * @param message The iterator to the message item the user clicked on or
	 *        info.data.end()
	 * @param time The time the user clicked on
	 */
	virtual void fillContextMenu(QMenu* menu, TapeChannelInfo& info,
	                             TapeChannelInfo::DataMap::iterator& message,
	                             int64 time);
	/**
	 * Called by DataView when the user opens a context menu and selects an action.
	 * @param action The action the user selected
	 * @param info The channel info for the channel the user clicked on
	 * @param message The iterator to the message item the user clicked on or
	 *        info.data.end()
	 * @param time The time the user clicked on
	 */
	virtual void executeAction(QAction* action, TapeChannelInfo& info,
	                           TapeChannelInfo::DataMap::iterator& message,
	                           int64 time);

	/// Should be overloaded in derived classes to provide the correct width in relation to the set height
	virtual int getDataWidth() const { return 1; }
	virtual int getDataHeight() const { return mDataHeight; }
	virtual int getIndicatorHeight() const { return mIndicatorHeight; }
	virtual int getTotalHeight() const { return mIndicatorHeight + mDataHeight; }

	virtual void setTotalHeight(int height) { mDataHeight = height-mIndicatorHeight; }

	virtual Typename getDatatype() { return ""; }
	virtual QSize renderMessage(QPainter* painter, const QRect& maxRect,
	                            TapeChannelInfo& info,
	                            TapeChannelInfo::DataMap::iterator& message);

	/**
	 * Called to get an information string about the given message in channel info.
	 * e.g. The data represented as string.
	 */
	virtual QString getMessageInfo(TapeChannelInfo& info,
	                               TapeChannelInfo::DataMap::iterator& message);

	/**
	 * Is called to add a new message at given time to channel info.
	 */
	virtual void addMessage(TapeChannelInfo& info, int64 time)
	{
	}

	/**
	 * Is called to add a new message at given time to channel info.
	 */
	virtual void editMessage(TapeChannelInfo& info,
	                         TapeChannelInfo::DataMap::iterator& message);

protected:

	TapeEditor* mEditor;
	int mDataHeight;
	int mIndicatorHeight;
	PluginFeature mFeatures;
};

typedef boost::shared_ptr<TapeDataRenderer> TapeDataRendererPtr;

///////////////////////////////////////////////////////////////////////////////

}

#endif 
