/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCConsole.C
 *    Console that allows to launch RPC calls easily
 *
 * @author Erik Einhorn
 * @date   2012/09/13
 */

#include <serialization/Serialization.h>

// include the view base class
#include <views/RPCConsole.h>

#include <boost/algorithm/string/trim.hpp>

// include Qt specific stuff
#include <QDir>
#include <QPushButton>
#include <QApplication>
#include <QHBoxLayout>

#include <widgets/QConsole.h>

#include <fw/Framework.h>
#include <rpc/RPCManager.h>

using namespace mira;

namespace mira { 

///////////////////////////////////////////////////////////////////////////////

class RPCConsole::Console : public QConsole
{
public:
	Console(QWidget* parent=NULL) :
		QConsole(parent,
		         "Enter your RPC calls or type 'help' for more information."),
		mWaitingForResult(false), mCompactPrint(false)
	{
		QFont font("Monospace");
		font.setStyleHint(QFont::TypeWriter);
		font.setPointSize(font.pointSize()-2);
		setCurrentFont(font);
		append(""); // add a new paragraph with these font settings
		setPrompt(">> ",true);
	}

	virtual bool isCommandComplete(const QString& cmd)
	{
		QString command = cmd.trimmed();
		int open  = command.count('(');
		int close = command.count(')');
		int quotes = command.count('"');
		return (open==close && quotes%2==0) || command.endsWith(';');
	}

	virtual QString interpretCommand(const QString& cmd, int *res) {
		QConsole::interpretCommand(cmd,res);

		QString command = cmd.trimmed();
		if(command.endsWith(';'))
			command.truncate(command.size()-1);

		if(command=="history clear") {
			history.clear();
			historyIndex = 0;
			return "";
		}
		if(command=="history") {
			int i=1;
			foreach(const QString& s, history)
				println(QString("    %1: ").arg(i++,2) + s);
			return "";
		}

		if(command=="trace") {
			println(QString::fromStdString(lastTrace));
			return "";
		}

		if(command.startsWith("compact")) {
			command = command.mid(7).trimmed();
			if (command == "on")
				mCompactPrint = true;
			else if (command == "off")
				mCompactPrint = false;
			else {
				*res = 1;
				return "'compact' must be followed by 'on' or 'off'";
			}

			return "";
		}

		if(command.startsWith("batch")) {
			auto ret = batch(command.mid(5).trimmed());
			*res = ret.first ? 0 : 1;
			return ret.second;
		}

		if(command.startsWith("help")) {
			command = command.mid(4).trimmed();
			help(command);
			return "";
		}

		auto ret = executeRPC(command);
		*res = ret.first ? 0 : 1;
		return ret.second;
	}

	std::pair<bool,QString> executeRPC(const QString& command)
	{
		int dotIdx = command.indexOf('.');
		if(dotIdx<0)
			return std::make_pair(false, QString("Missing service or method! A valid call has the format: service.method(parameters)"));

		QString service = command.mid(0,dotIdx);
		QString rest = command.mid(dotIdx+1);
		QString method, params;

		int paramBegin = rest.indexOf('(');
		if(paramBegin<0) {
			// no params
			method = rest;
		} else {
			// extract params:
			int paramEnd = rest.lastIndexOf(')');
			method = rest.mid(0, paramBegin);
			params = rest.mid(paramBegin+1, paramEnd-paramBegin-1);
		}

		try {
			mWaitingForResult = true;
			auto future = MIRA_FW.getRPCManager().callJSON(service.toStdString(),
			                                               method.toStdString(),
			                                               params.toStdString());

			while(mWaitingForResult) {
				if(future.timedWait(Duration::milliseconds(10))) {
					JSONValue v = future.get();
					std::string r;
					if(!v.is_null())
						r = json::write(future.get(), !mCompactPrint);
					mWaitingForResult = false;
					return std::make_pair(true, QString::fromStdString(r));
				}
				QApplication::processEvents();
			}

			// we reach here if we aborted (mWaitingForResult was set to false)
			return std::make_pair(false, QString("Aborted waiting for result"));

		} catch(Exception& ex) {
			std::ostringstream ss;
			ss << ex.callStack();
			lastTrace = ss.str();
			mWaitingForResult = false;
			return std::make_pair(false, QString::fromStdString(ex.message()));
		}
	}

	void abortWaiting()
	{
		mWaitingForResult = false;
	}

	//give suggestions to autocomplete a command
	virtual QStringList suggestCommand(const QString& cmd, QString& prefix)
	{
		QString command = cmd.trimmed();
		// handle batch
		if(command.startsWith("batch")) {
			QStringList list = suggestFiles(command.mid(5).trimmed(), prefix);
			prefix = "batch " + prefix;
			return list;
		}

		// handle help
		if(command.startsWith("help")) {
			prefix = "help ";
			command = command.mid(4).trimmed();
		}

		if(!command.contains('.')) {
			// command does not contain a '.' yet, so list the services

			std::set<std::string> allServices = MIRA_FW.getRPCManager().getLocalServices();
			std::set<std::string> remoteServices = MIRA_FW.getRPCManager().getRemoteServices();
			allServices.insert(remoteServices.begin(), remoteServices.end());

			QStringList serviceList;
			foreach(const std::string& s, allServices)
			{
				QString service = QString::fromStdString(s);
				if(service.startsWith(command, Qt::CaseInsensitive)) {
					// filter hidden services (unless the user shows them by typing the next #)
					if(service.indexOf('#',command.size()+1)==-1)
						serviceList.append(QString::fromStdString(s));
				}
			}

			// if nothing was found above, then search for any occurrence of command
			if(serviceList.empty()) {
				foreach(const std::string& s, allServices)
				{
					QString service = QString::fromStdString(s);
					if(service.contains(command, Qt::CaseInsensitive))
						serviceList.append(QString::fromStdString(s));
				}
			}

			return serviceList;
		} else {
			if(!command.contains('(')) {
				// command already contains a '.' and not a '(' yet, so list the methods
				int dotIdx = command.indexOf('.');
				assert(dotIdx>0);
				QString service = command.mid(0,dotIdx);
				QString methodprefix = command.mid(dotIdx+1);

				try {
					auto s = MIRA_FW.getRPCManager().getLocalService(service.toStdString());
					prefix += command.mid(0,dotIdx+1);
					return filterMethods(s, methodprefix);
				} catch(...){}

				try {
					auto s = MIRA_FW.getRPCManager().getRemoteService(service.toStdString());
					prefix += command.mid(0,dotIdx+1);
					return filterMethods(s, methodprefix);
				} catch(...){}
			} else {
				// command already contains a '('
				if (command.count('"') % 2 != 0) {
					// open quote -> entering a string parameter (possibly a file path)
					int quoteIdx = command.lastIndexOf('"');
					QStringList list = suggestFiles(command.mid(quoteIdx+1).trimmed(), prefix);
					prefix = command.mid(0, quoteIdx+1) + prefix;
					return list;
				}
			}
		}

		// nothing was found, so suggest original command
		return QStringList() << command;
	}


	template <typename Service>
	QStringList filterMethods(const Service& service, const QString& prefix)
	{
		QStringList list;
		foreach(const auto& m, service.methods)
		{
			QString method = QString::fromStdString(m.signature.name);
			if(method.startsWith(prefix, Qt::CaseInsensitive))
				list.append(method);
		}

		// if still empty, then search for any occurrence
		if(list.empty()) {
			foreach(const auto& m, service.methods)
			{
				QString method = QString::fromStdString(m.signature.name);
				if(method.contains(prefix, Qt::CaseInsensitive))
					list.append(method);
			}
		}

		return list;
	}

	std::pair<bool,QString> batch(const QString& file)
	{
		std::ifstream s(file.toStdString().c_str());
		if(!s.is_open())
			return std::make_pair(false, QString("Failed to open '")+file+"'");

		while(!s.eof()) {
			std::string line;
			std::getline(s, line);

			boost::algorithm::trim(line);

			if(line.empty())
				continue;

			if(line[0]=='#') // skip comments
				continue;

			setTextColor(Qt::darkGray);
			println("> " + line);

			auto ret = executeRPC(QString::fromStdString(line));
			if(!ret.first)
				return ret;

			if(!ret.second.isEmpty()) {
				setTextColor(outColor_);
				println(ret.second);
			}
		}

		return std::make_pair(true, QString());
	}

	QStringList suggestFiles(const QString& path,  QString& prefix)
	{
		int lastSlash=path.lastIndexOf("/");
		QString suffix = path;

		if(lastSlash>=0) {
			prefix = path.mid(0,lastSlash) + "/";
			suffix = path.mid(lastSlash+1);
		}

		QDir dir(prefix);

		QStringList list;
		QStringList entries = dir.entryList(QDir::AllDirs);
		foreach(const QString& e, entries)
			if(e.startsWith(suffix,Qt::CaseInsensitive))
				list.push_back(e+"/");

		entries = dir.entryList(QDir::Files);
		foreach(const QString& e, entries)
			if(e.startsWith(suffix,Qt::CaseInsensitive))
				list.push_back(e);

		return list;
	}

	void help(QString command)
	{
		command = command.trimmed();

		// list all services, if help cmd is empty
		if(command.isEmpty()) {
			std::set<std::string> allServices = MIRA_FW.getRPCManager().getLocalServices();
			std::set<std::string> remoteServices = MIRA_FW.getRPCManager().getRemoteServices();
			allServices.insert(remoteServices.begin(), remoteServices.end());
			println("    Type the service name followed by the method and parameters to execute an RPC call, e.g.:");
			println("      myservice.foo(\"string\", 123, 456.78, {\"X\":1,\"Y\":2,\"Phi\":90})");
			println("");
			println("    Available commands:");
			println("      help <service>           Get information to a service, including all available methods");
			println("      help <service>.<method>  Get detailed information to a method of a service.");
			println("      history                  List the previously entered commands.");
			println("      history clear            Clear the command history");
			println("      trace                    Show stack trace for last error");
			println("      compact on/off           Switch between formatted and compact display of JSON results");
			println("      batch <file>             Runs all commands listed in the given script/batch file");
			println("");
			println("    Key commands:");
			println("      <Tab>                    Auto-completion for service names and methods");
			println("      <Arrow Up/Down>          Step through the command history");
			println("      <Ctrl>+R                 Search command history");
			println("      <Ctrl>+M                 Search method names over all services");
			println("      <Ctrl>+C                 Abort waiting for the result of an RPC call, if the call takes\n"
			        "                               too long. The execution of the call itself, however, will\n"
			        "                               continue in background.");
			println("");
			println("    Available services:");
			foreach(const std::string& s, allServices)
			{
				QString service = QString::fromStdString(s);
				if(!service.contains('#'))
					println("      " + service);
			}
			return;
		}

		QString service = command;
		QString methodprefix;

		if(command.contains('.')) {
			// command already contains a '.', extract service and method name
			int dotIdx = command.indexOf('.');
			assert(dotIdx>0);
			service = command.mid(0,dotIdx);
			methodprefix = command.mid(dotIdx+1);
		}

		try {
			auto s = MIRA_FW.getRPCManager().getLocalService(service.toStdString());
			helpMethod(s, methodprefix);
			return;
		} catch(...){}

		try {
			auto s = MIRA_FW.getRPCManager().getRemoteService(service.toStdString());
			helpMethod(s, methodprefix);
			return;
		} catch(...){}

		println("No such service.");
	}


	template <typename Service>
	void helpMethod(const Service& service, const QString& methodprefix)
	{
		bool hasmethod=false;
		foreach(const auto& m, service.methods)
		{
			QString method = QString::fromStdString(m.signature.name);
			if(methodprefix.isEmpty() || method==methodprefix)
			{
				setFormat(true);
				println("    " +  QString::fromStdString(m.extendedSignature()));
				setFormat(false,true);
				println("      " + m.comment);
				if (method == methodprefix) {
					if (!m.parameterDesc.empty())
						println(QString::fromStdString(m.parameterDescriptions("      ")));
					println("      Usage: " + m.signature.name+"("+m.sampleParametersSet()+")"
					        + (m.parameterSamplesDefault && !m.parameterSamples.empty() ?
					        " (sample arguments for this method default-generated)" : ""));
				}
				hasmethod=true;
			}
		}
		if(!hasmethod)
			println("No such service method.");
	}


	std::list<std::string> getHistory() const
	{
		std::list<std::string> h;
		foreach(const QString& s, history)
			h.push_back(s.toStdString());
		return h;
	}

	void setHistory(const std::list<std::string>& h)
	{
		history.clear();
		foreach(const std::string& s, h)
			history.append(QString::fromStdString(s));
		historyIndex=history.size();
	}

protected:

	void keyPressEvent(QKeyEvent * e)
	{
		if ( (e->modifiers() & Qt::ControlModifier) && (e->key() == Qt::Key_C) )
		{
			abortWaiting();
			return;
		}

		if (mWaitingForResult) // ignore any other key events while waiting for RPC result,
			return;        // prevents calling the command again or changing the line content

		QConsole::keyPressEvent(e);
	}

	MIRA_EXTENSIBLE_ENUM_DECLARE(SearchMode, QConsole::SearchMode,
	                             SEARCH_METHODS)

	virtual const QConsole::SearchMode& searchModeCommand(QKeyEvent* e)
	{
		if((e->modifiers() & Qt::ControlModifier) && (e->key() == Qt::Key_M))
			return SearchMode::SEARCH_METHODS;

		return QConsole::searchModeCommand(e);
	}

	virtual void enterSearchMode(const QConsole::SearchMode& mode)
	{
		QConsole::enterSearchMode(mode);

		if (mode != SearchMode::SEARCH_METHODS)
			return;

		allMethods.clear();

		std::set<std::string> allServices = MIRA_FW.getRPCManager().getLocalServices();
		std::set<std::string> remoteServices = MIRA_FW.getRPCManager().getRemoteServices();
		allServices.insert(remoteServices.begin(), remoteServices.end());

		foreach (const auto& service, allServices)
		{
			try {
				auto s = MIRA_FW.getRPCManager().getLocalService(service);
				foreach(const auto& m, s.methods)
					allMethods.insert(std::make_pair(QString::fromStdString(m.signature.name),
					                                 QString::fromStdString(service+"."+m.signature.name)));
			} catch(...){}

			try {
				auto s = MIRA_FW.getRPCManager().getRemoteService(service);
				foreach(const auto& m, s.methods)
				allMethods.insert(std::make_pair(QString::fromStdString(m.signature.name),
				                                 QString::fromStdString(service+"."+m.signature.name)));
			} catch(...){}
		}
	}

	virtual QString getSearchModeDescriptor()
	{
		if (currentSearchMode == SearchMode::SEARCH_METHODS)
			return "method-search";

		return QConsole::getSearchModeDescriptor();
	}

	virtual QStringList getSearchList()
	{
		if (currentSearchMode == SearchMode::SEARCH_METHODS)
			return QStringList();

		return QConsole::getSearchList();
	}

	virtual std::multimap<QString, QString> getExtendedSearchList()
	{
		if (currentSearchMode == SearchMode::SEARCH_METHODS)
			return allMethods;

		return QConsole::getExtendedSearchList();
	}

private:

	bool mWaitingForResult;

	std::multimap<QString, QString> allMethods;

	bool mCompactPrint;
	std::string lastTrace;
};

MIRA_EXTENSIBLE_ENUM_DEFINE(RPCConsole::Console::SearchMode, SEARCH_METHODS)

///////////////////////////////////////////////////////////////////////////////

RPCConsole::RPCConsole() : mConsole(NULL)
{
	connect(qApp, SIGNAL(lastWindowClosed()), this, SLOT(onMIRACenterClosed()));
}

QWidget* RPCConsole::createPartControl()
{
	QWidget* w = new QWidget(this);
	QHBoxLayout* l = new QHBoxLayout(w);
	l->setContentsMargins(0,3,0,0);

	mConsole = new Console(w);
	l->addWidget(mConsole);
	mConsole->setHistory(mHistory);

	return w;
}

const std::list<std::string>& RPCConsole::getHistory()
{
	if(mConsole)
		mHistory = mConsole->getHistory();
	return mHistory;
}

void RPCConsole::setHistory(const std::list<std::string>& history)
{
	mHistory = history;
	if(mConsole)
		mConsole->setHistory(mHistory);
}

void RPCConsole::closeEvent(QCloseEvent* event)
{
	mConsole->abortWaiting();
	ViewPart::closeEvent(event);
}

void RPCConsole::onMIRACenterClosed()
{
	mConsole->abortWaiting();
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::RPCConsole, mira::ViewPart );
