/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IMotorController.h
 *    An interface class for objects, which provide a motor controller.
 *
 * @author Tim Langner
 * @date   2012/10/10
 */

#ifndef _MIRA_IMOTORCONTROLLER_H_
#define _MIRA_IMOTORCONTROLLER_H_

#include <robot/IDrive.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/// An interface class for objects, which provide a motor controller.
class IMotorController : public IDrive
{
public:
	/** @name Destructor and reflection */
	//@{

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, IDrive);

		r.interface("IMotorController");
		r.method("enableMotors", &IMotorController::enableMotors, this,
		         "Enables/disable the motors.",
		         "enable", "whether to enable", true);
		r.method("resetMotorStop", &IMotorController::resetMotorStop, this,
		         "Reset the motor stop flags (bumper, emergency stop flags, etc).");
		r.method("suspendBumper", &IMotorController::suspendBumper, this,
		         "Temporarily suspend bumper detection.");
		r.method("suspendBumperSupported", &IMotorController::suspendBumperSupported, this,
		         "Returns true if suspending the bumper is supported by the motor controller.");
		r.method("resetOdometry", &IMotorController::resetOdometry, this,
		         "Reset the odometry to zero.");
		r.method("emergencyStop", &IMotorController::emergencyStop, this,
		         "Performs an emergency stop and sets the motor emergency stop flag.");
	}

	virtual ~IMotorController() {}

	//@}

public:
	/** @name IMotorController interface definition */
	//@{

	/**
	 * Enables/disable the motors.
	 * If the motors are disabled, the drive system should be in a idle mode.
	 * That means, that the robot should be movable by hand ("free run"), i.e.
	 * the motors must be turned off.
	 * @param[in] enable The enable/disable flag.
	 */
	virtual void enableMotors(bool enable) = 0;

	/// Reset the motor stop flags (bumper, emergency stop flags, etc).
	virtual void resetMotorStop() = 0;

	/// Temporarily disable Bumper.
	virtual void suspendBumper() {}

	/// Whether the motor controller can suspend (temporarily) the bumper detection
	virtual bool suspendBumperSupported()
	{
		return false;
	}

	/// Reset the odometry to zero.
	virtual void resetOdometry() = 0;

	/**
	 * Performs an emergency stop and sets the motor emergency stop flag.
	 * Hardware, that does not support an emergency stop explicitly must
	 * set the velocity to 0 and make sure that no other velocity can be set
	 * until resetMotorStop() is called.
	 */
	virtual void emergencyStop() = 0;

	//@}
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
