/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DifferentialProbabilisticMotionModel.h
 *    Probabilistic motion model for a robot with differential drive.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2009/08/03
 */

#ifndef _MIRA_DIFFERENTIALPROBABILISTICMOTIONMODEL_H_
#define _MIRA_DIFFERENTIALPROBABILISTICMOTIONMODEL_H_

#include <math/Random.h>

#include <robot/ProbabilisticMotionModel.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Probabilistic motion model for a robot with differential drive.
 * (2 wheels on an axis, fixed orientation).
 *
 * This class represents the probability density on two different ways:
 * - it provides methods to sample from the probability density in order to
 *   represent it using particles. This is useful when the motion model is
 *   used with particle filters
 * - it allows to represent the probability as normal distribution. The
 *   distribution can be updated using the gaussianMotion() method.
 */
class DifferentialProbabilisticMotionModel : public ProbabilisticMotionModel
{
public:
	/**
	 * A struct which is used by @see calculateMotionParameters() and
	 * @see sampleMotion() to store motion parameters to gain a better
	 * performance. The values for the parameters in this model are based
	 * on the motion model from Thrun, Burgard, Fox in Probabilistic Robotics.
	 * (@see http://www.mrpt.org/Probabilistic_Motion_Models#References).
	 */
	struct MotionParameters
	{
		MotionParameters() :
			trans(0.0f), rot1(0.0f), rot2(0.0f),
			transStddev(0.0f), rot1Stddev(0.0f), rot2Stddev(0.0f) {}

		// means
		float trans;
		float rot1;
		float rot2;

		// std.devs
		float transStddev;
		float rot1Stddev;
		float rot2Stddev;
	};

public:
	/** @name Constructors and reflect */
	//@{

	DifferentialProbabilisticMotionModel();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("AlphaRotRot", alphaRotRot,
		           "Alpha for rotation drift while rotating", 0.2f);
		r.property("AlphaRotTrans", alphaRotTrans,
		           "Alpha for translation drift while rotating", 0.2f);
		r.property("AlphaTransTrans", alphaTransTrans,
		           "Alpha for translation drift while translating", 0.2f);
		r.property("AlphaTransRot", alphaTransRot,
		           "Alpha for rotation drift while translating", 0.2f);
	}

	//@}

public:

	/**
	 * Calculate the motion model parameters.
	 *
	 * This method computes the motion parameters for odometric drift based on
	 * the odometry motion model from Thrun, Burgard, Fox in "Probabilistic
	 * Robotics", 2005.
	 * @see http://www.mrpt.org/Probabilistic_Motion_Models#References
	 *
	 * @code
	 *   Y                        /         _
	 *   ^                       /\ rot2_ -
	 *   |                      /  |_ -
	 *   |                     /_ -
	 *   |                  _ -P1
	 *   |              _ -
	 *   |    trans _ - \
	 *   |      _ -      | rot1
	 *   |  _ -          |
	 * P0............... /...........................> X
	 * @endcode
	 *
	 * x,y,phi denote the last estimated pose P0 at t-1 and x',y',phi' the new
	 * estimated pose P1 at t. Compute the relative motion parameters:
	 *
	 * trans = sqrt((x-x')^2+(y-y')^2)
	 * rot1 = atan2(y'-y, x'-x) - phi
	 * rot2 = phi' - phi - d_rot1
	 *
	 * If (x,y) and (x',y') are very nearby (< 0.005m), rot1 is set to zero.
	 *
	 * transStddev = alphaTransTrans * trans + alphaTransRot * (rot1 + rot2)
	 * rot1Stddev  = alphaRotRot * rot1 + alphaRotTrans * trans
	 * rot2Stddev  = alphaRotRot * rot2 + alphaRotTrans * trans
	 *
	 * @param[in] p0 The last observed pose
	 * @param[in] p1 The current observed pose
	 * @return The computed motion parameters.
	 */
	MotionParameters calculateMotionParameters(const Pose2& p0, const Pose2& p1) const;


	/** @name Implementation of ProbabilisticMotionModel*/
	//@{
	virtual Pose2 sampleMotion(const Pose2& startPose, const Pose2& p0, const Pose2& p1) const;

	virtual void sampleMotion(const std::vector<Pose2>& startPoses, const Pose2& p0, const Pose2& p1,
	                          std::vector<Pose2>& sampledPoses) const;
	//@}

	/**
	 * Samples motion using parameters for odometric drift and the odometry
	 * motion model from Thrun, Burgard, Fox in Probabilistic Robotics.
	 *
	 * This method takes motion model parameters that can be calculate by the
	 * method calculateMotionParameters().
	 * A new pose is sampled based on the starting pose startPose.
	 *
	 * Calculation of a motion sample:
	 *
	 * trans^ = trans - N(0, transStddev * transStddev)
	 * rot1^  = rot1  - N(0, rot1Stddev * rot1Stddev)
	 * rot2^  = rot2  - N(0, rot2Stddev * rot2Stddev)
	 *
	 * x   = trans^ * cos(startPhi + rot1^)
	 * y   = trans^ * sin(startPhi + rot1^)
	 * phi = rot1^ + rot2^
	 *
	 * The sampled motion can be used to adapt a pose of a particle:
	 *
	 * x'   = startX + x
	 * y'   = startY + y
	 * phi' = startPhi + phi
	 *
	 * If parameters are known and many motion samples need to be computed
	 * (e.g for a particle filter) this method can be used to save some
	 * performance.
	 * @param[in] startPose  The starting position for applying the sampled motion
	 * @param[in] parameters The motion model parameters
	 *                       (can be obtained via calculateMotionParameters())
	 * @return Starting position updated by the sampled motion
	 */
	Pose2 sampleMotion(const Pose2& startPose, const MotionParameters& parameters) const;

	/**
	 * Applies a gaussian motion model to the initial pose at startPose.
	 *
	 * @param[in] startPose  The starting position for applying the sampled motion
	 * @param[in] parameters The motion model parameters
	 *                       (can be obtained via calculateMotionParameters())
	 * @return Returns the new robot pose with the updated covariance.
	 */
	PoseCov2 gaussianMotion(const PoseCov2& startPose, const MotionParameters& parameters) const;

public:

	/// Odometry motion parameters for modeling drift @see sampleMotion()
	float alphaRotRot;
	float alphaRotTrans;
	float alphaTransTrans;
	float alphaTransRot;
};

///////////////////////////////////////////////////////////////////////////////

inline Pose2 DifferentialProbabilisticMotionModel::sampleMotion(const Pose2& startPose,
                                                                const MotionParameters& parameters) const
{
	float dTransSample = parameters.trans - (float)MIRA_RANDOM.normal(parameters.transStddev);
	float dRot1Sample = smallestAngleDifference(parameters.rot1,
	                                            (float)MIRA_RANDOM.normal(parameters.rot1Stddev));
	float dRot2Sample = smallestAngleDifference(parameters.rot2,
	                                            (float)MIRA_RANDOM.normal(parameters.rot2Stddev));

	return Pose2(startPose.x() + dTransSample * std::cos(startPose.phi() + dRot1Sample),
	             startPose.y() + dTransSample * std::sin(startPose.phi() + dRot1Sample),
	             startPose.phi() + SignedAnglef(dRot1Sample + dRot2Sample).rad());

}

///////////////////////////////////////////////////////////////////////////////

}}

#endif
