/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CollisionTest.C
 *    Test cases for collision
 *
 * @author Tim Langner
 * @date   2012/04/03
 */

#include <boost/test/unit_test.hpp>

#include <model/CollisionTest.h>
#include <model/RigidModel.h>

#include <opencv2/imgproc/imgproc.hpp>

using namespace mira;

const char* argv[] = {"RigidModelCollisionTest", "-d2", "--no-colors"};
Framework fw(3,(char**)argv);

#if CV_MAJOR_VERSION >= 4
	#define CV_FILLED cv::FILLED
#endif

BOOST_AUTO_TEST_CASE(TestCollision)
{
	// generate a model with a single link and a single joint
	// the model is 6mx1m in size and shifted 1.5m to the side (positive y = left)
	// so it rotates about a center that is 1.5m to the model's right side
	model::RigidModel m;
	model::LinkPtr link(new model::Link());
	link->name = "Link1";
	link->collisions.push_back(model::CollisionPtr(new model::Collision()));
	link->collisions.front()->origin.y() = 1.5f;
	model::BoxPtr box(new model::Box());
	box->size = Size3f(6, 1, 1);
	link->collisions.front()->geometry = box;
	m.links.push_back(link);

	model::JointPtr joint(new model::Joint());
	joint->parent = "Root";
	joint->child = "Link1";
	m.joints["root"] = joint;

	m.resolveAndPublishLinks("/");

	// create and initialize a collision test object
	float cellSize = 0.01f;
	model::CollisionTest ct2(cellSize);
	ct2.initialize(m.getFootprint("/Root"));

	// Test copying CollisionTest object!
	model::CollisionTest ct(ct2);

	// create a 10mx10m collision map and draw a 5mx5m obstacle in the upper left corner
	Point2i center((int)(5.0f / cellSize), (int)(5.0f / cellSize));
	Img8U1 map((int)(10.0f / cellSize), (int)(10.0f / cellSize));
	map = 0;
	cv::rectangle((cv::Mat&)map, Point2i(0, 0), center, 255, CV_FILLED);
	// calculate distance map for distance to obstacle computations
	Img32F1 distanceMap;
	ct.distanceTransform(map, distanceMap);

	// there should be no collision between the model's footprint
	// and the map for 270° <= d <= 360° 
	for (float d = 270.0f; d<=360.0f; d+=10.0f)
	{
		BOOST_CHECK(!ct.testCollision(map, center, deg2rad(d)));
		float distance = ct.distanceToObstacle(distanceMap, center, deg2rad(d)) * cellSize;
		std::cout << d << ":" << distance << std::endl;
		// TODO check why this is not always in the range of 1.0f +- 2*cellSize
		// BOOST_CHECK_SMALL(distance - 1.0f, cellSize*2.0f);
	}

	// there should be a collision between the model's footprint
	// and the map for 90° <= d <= 180° 
	for (float d = 90.0f; d<=180.0f; d+=10.0f)
	{
		BOOST_CHECK(ct.testCollision(map, center, deg2rad(d)));
		// distance to obstacle must be 0.0f since we are colliding
		float distance = ct.distanceToObstacle(distanceMap, center, deg2rad(d)) * cellSize;
		BOOST_CHECK(distance == 0.0f);
	}

	Rect2f botBB = m.getFootprint("/Root").getBoundingBox();
	Rect2i botBBinMap(map.width()/2 + botBB.x0()/cellSize,
	                  map.height()/2 + botBB.y0()/cellSize,
	                  botBB.width()/cellSize, botBB.height()/cellSize);

	// create obstacles just beside the robot (not colliding)
	std::vector<Rect2i> rects;
	rects = {{0,                 0,                  botBBinMap.x0()-1,             map.height()},
	         {0,                 0,                  map.width(),                   botBBinMap.y0()-1},
	         {botBBinMap.x1()+1, 0,                  map.width()-botBBinMap.x1()-1, map.height()},
	         {0,                 botBBinMap.y1()+1,  map.width(),                   map.height()-botBBinMap.y1()-1}};
	foreach (const Rect2i& r, rects) {
		map = 0;
		cv::rectangle((cv::Mat&)map, r.minCorner, r.maxCorner, 255, CV_FILLED);
		ct.distanceTransform(map, distanceMap);
		BOOST_CHECK(!ct.testCollision(map, center, 0));
		float distance = ct.distanceToObstacle(distanceMap, center, 0);
		BOOST_CHECK_CLOSE(distance, 1.0, 50/*percent*/);
	}

	// create obstacles just at the edge of the robot (colliding)
	rects = {{0,               0,               botBBinMap.x0(),             map.height()},
	         {0,               0,               map.width(),                 botBBinMap.y0()},
	         {botBBinMap.x1(), 0,               map.width()-botBBinMap.x1(), map.height()},
	         {0,               botBBinMap.y1(), map.width(),                 map.height()-botBBinMap.y1()}};
	foreach (const Rect2i& r, rects) {
		map = 0;
		cv::rectangle((cv::Mat&)map, r.minCorner, r.maxCorner, 255, CV_FILLED);
		ct.distanceTransform(map, distanceMap);
		BOOST_CHECK(ct.testCollision(map, center, 0));
		BOOST_CHECK_EQUAL(ct.distanceToObstacle(distanceMap, center, 0), 0.f);
	}

}

BOOST_AUTO_TEST_CASE(DecayShape)
{
	Polygon2f rect; // rect 2mx1m
	rect.emplace_back(-1.f, -0.5f);
	rect.emplace_back( 1.f, -0.5f);
	rect.emplace_back( 1.f,  0.5f);
	rect.emplace_back(-1.f,  0.5f);
	Footprint fp;
	fp.push_back(rect);

	float cellSize = 0.01f;
	model::CollisionTest ct(cellSize);
	ct.initialize(fp);

	// create a 10mx10m map
	Img8U1 map((int)(10.0f / cellSize), (int)(10.0f / cellSize));
	int w = map.width();
	int h = map.height();

	// fill upper half with 20, lower half with 200
	map = 200;
	cv::rectangle((cv::Mat&)map, Point2i(0, 0), Point2i(w,h/2), 20, CV_FILLED);

	ct.decayShape(map, Point2i(w/2,h/2), 0.f,
	              /*decayRate=*/0.1f, /*neutral=*/127.f,
	              /*decayObstacles=*/true, /*decayFreespace=*/false,
	              /*invertShape=*/ false);

	BOOST_CHECK_EQUAL(map(0,0), 20);        // outside shape unchanged
	BOOST_CHECK_EQUAL(map(0,h-1), 200);     // outside shape unchanged

	BOOST_CHECK_EQUAL(map(w/2,h/2-1), 20);  // free space not decayed
	BOOST_CHECK_EQUAL(map(w/2,h/2+1), 192); // obstacle decayed, d = (200-127)/10

	ct.decayShape(map, Point2i(w/2,h/2), 0.f,
	              /*decayRate=*/0.1f, /*neutral=*/127.f,
	              /*decayObstacles=*/false, /*decayFreespace=*/true,
	              /*invertShape=*/ false);

	BOOST_CHECK_EQUAL(map(w/2,h/2-1), 31);  // free space decayed, d = (127-20)/10
	BOOST_CHECK_EQUAL(map(w/2,h/2+1), 192); // obstacle not decayed further

	ct.decayShape(map, Point2i(w/2,h/2), 0.f,
	              /*decayRate=*/0.2f, /*neutral=*/100.f,
	              /*decayObstacles=*/true, /*decayFreespace=*/true,
	              /*invertShape=*/ false);

	BOOST_CHECK_EQUAL(map(w/2,h/2-1), 45);  // d = (100-31)/5
	BOOST_CHECK_EQUAL(map(w/2,h/2+1), 173); // d = (192-100)/5


	// now decay the inverted shape 
	// --> upper/lower half should have uniform values

	ct.decayShape(map, Point2i(w/2,h/2), 0.f,
	              /*decayRate=*/0.1f, /*neutral=*/127.f,
	              /*decayObstacles=*/true, /*decayFreespace=*/true,
	              /*invertShape=*/ true);
	ct.decayShape(map, Point2i(w/2,h/2), 0.f,
	              /*decayRate=*/0.2f, /*neutral=*/100.f,
	              /*decayObstacles=*/true, /*decayFreespace=*/true,
	              /*invertShape=*/ true);

	for (int y = 0; y <= h/2; ++y)
		for (int x = 0; x < w; ++x)
			BOOST_CHECK_EQUAL(map(x,y), 45);

	for (int y = h/2+1; y < h; ++y)
		for (int x = 0; x < w; ++x)
			BOOST_CHECK_EQUAL(map(x,y), 173);
}

