/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RigidModelVisualization2D.C
 *    Visualization for rigid models in the 2D view.
 *
 * @author Christian Reuther
 * @date   2014/09/03
 */

#include <visualization/Visualization2D.h>

#include <model/RigidModel.h>
#include <model/RigidModelObserver.h>
#include <transform/Pose.h>
#include <fw/TransformProperty.h>
#include <fw/ServiceProperty.h>

#include <serialization/Serialization.h>
#include <serialization/adapters/Qt/QColor>

#include <QGraphicsScene>
#include <QGraphicsPathItem>
#include <QGraphicsEllipseItem>
#include <QGraphicsItemGroup>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

class RigidModelVisualization2D : public Visualization2D {
	MIRA_META_OBJECT(RigidModelVisualization2D,
				("Name", "Rigid Model")
				("Description", "2D visualization of the rigid model's footprint")
				("Category", "Models"))
public:
	RigidModelVisualization2D() : mModelProvider("IRigidModelProvider") {
		mShowFootprint = true;
		mShowFootprintCircles = false;
		mFillFootprint = true;
		mShowXAxis = true;
		mFootprintColor = QColor(175, 0, 0, 175);
		mHaveFootprint = false;

		mInnerCircleItem = NULL;
		mOuterCircleItem = NULL;
		mFootprintItem = NULL;
		mItem = NULL;
		mXAxisItem = NULL;
	}

	~RigidModelVisualization2D() {
		if(!getSite() || !getSite()->getSceneManager())
			return;

		clearObjects();
		getSite()->getSceneManager()->removeItem(mItem);
		delete mItem;
	}

	template <typename Reflector> void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Visualization2D);
		r.property("ModelProvider", mModelProvider, "The service that provides us with a rigid model");
		r.property("Show Footprint", mShowFootprint, setter<bool>(&RigidModelVisualization2D::setShowFootprint, this),
		           "Toggle visualization of footprint", true);
		r.property("Show Footprint Circles", mShowFootprintCircles, setter<bool>(&RigidModelVisualization2D::setShowFootprintCircles, this),
		           "Toggle visualization of the outer and inner circles (encircle and incircle)", false);
		r.property("Fill footprint", mFillFootprint, setter<bool>(&RigidModelVisualization2D::setFillFootprint, this),
		           "Toggle drawing an outline of (false) or a fully filled (true) footprint.", true);
		r.property("Footprint color", mFootprintColor, setter<QColor>(&RigidModelVisualization2D::setFootprintColor, this),
		           "Set the color of either the footprint outline or the filled footprint", QColor(175, 0, 0, 175));
		r.property("ShowXAxis", mShowXAxis, setter<bool>(&RigidModelVisualization2D::setShowXAxis, this),
			"Toggle drawing a line to indicate the positive X axis of the robot coordinate frame", true);
	}

	virtual void setupScene(IVisualization2DSite* site) {
		mItem = new QGraphicsItemGroup();
		site->getSceneManager()->addItem(mItem);
	}

	virtual QGraphicsItem* getItem() {
		return mItem;
	}

	virtual void setEnabled(bool enabled) {
		Visualization2D::setEnabled(enabled);
		mItem->setVisible(enabled);
	}

	void setShowFootprint(bool show) {
		mShowFootprint = show;
		updateObjects();
	}

	void setFillFootprint(bool fill) {
		mFillFootprint = fill;
		updateObjects();
	}

	void setFootprintColor(QColor color) {
		mFootprintColor = color;
		updateObjects();
	}

	void setShowFootprintCircles(bool show) {
		mShowFootprintCircles = show;
		updateObjects();
	}

	void setShowXAxis(bool show) {
		mShowXAxis = show;
		updateObjects();
	}

protected:
	virtual void update(Duration dt);
	bool updateProvider();
	bool updateFootprints();

	void clearObjects();
	void createObjects();
	void updateObjects();

	void collisionChanged();

protected:
	ServiceProperty mModelProvider;
	std::string mUsedModelProvider;
	std::string mRootLinkFrame;

	RigidModelPtr mModel;
	RigidModelObserverPtr mModelObserver;

	bool mShowFootprint;
	bool mShowFootprintCircles;
	bool mFillFootprint;
	bool mShowXAxis;

	QGraphicsItemGroup* mItem;
	QGraphicsPathItem* mFootprintItem;
	QGraphicsEllipseItem* mOuterCircleItem;
	QGraphicsEllipseItem* mInnerCircleItem;
	QGraphicsLineItem* mXAxisItem;

	QColor mFootprintColor;
	bool mHaveFootprint;
};

///////////////////////////////////////////////////////////////////////////////

void RigidModelVisualization2D::update(Duration dt) {
	bool haveProvider = false;
	try {
		haveProvider = updateProvider();
	} catch(Exception& ex) {
		error("Model", ex.message(), "Error in model");
	}

	// We might not have our footprint yet, due to the transform tree not being available yet.
	if(!mHaveFootprint)
		mHaveFootprint = updateFootprints();

	// Without a model provider, we won't know our root link frame and can't transform the model
	// onto its correct position.
	if(!haveProvider || !mHaveFootprint)
		return;

	try {
		const std::string fixedFrame = this->getSite()->getFixedFrame();
		const std::string cameraFrame = this->getSite()->getCameraFrame();

		const Time now = Time::now();
		const Pose2 pose = getAuthority().getTransform<Pose2>(mRootLinkFrame, now, cameraFrame, now, fixedFrame);

		// To deal with the inverted y axis, we need to negate the y position and invert the rotation
		mItem->setPos(pose.x(), pose.y());
		mItem->setRotation(rad2deg<float>(pose.phi()));
	} catch(const Exception& ex) {
		MIRA_LOG(DEBUG) << "RigidModelVisualization2D: Error while transforming model into correct frame. Exception: " << ex.what();
	}
}

bool RigidModelVisualization2D::updateProvider() {
	std::string provider;
	if(!mModelProvider.empty()) {
		// same provider as already used and it still exists
		if(mUsedModelProvider == mModelProvider.getService() && getAuthority().existsService(mUsedModelProvider))
			return true;
		clearObjects();
		mUsedModelProvider = "";
		provider = mModelProvider;
	} else {
		// we already have a model provider and it still exists -> do nothing
		if(!mUsedModelProvider.empty() && getAuthority().existsService(mUsedModelProvider))
			return true;
		clearObjects();
		mUsedModelProvider = "";
		auto l = getAuthority().queryServicesForInterface("IRigidModelProvider");
		// no provider found
		if(l.empty())
			return false;
		provider = *l.begin();
	}

	auto p = getAuthority().callService<RigidModelPtr>(provider, "getRigidModel");
	if(!p.timedWait(Duration::seconds(1000)))
		MIRA_THROW(XRuntime, "Model could not be retrieved from service '" << provider << "'");
	mUsedModelProvider = provider;

	mModel = p.get();
	if(!mModel)
		MIRA_THROW(XRuntime, "Model retrieved from service '" << provider << "' is NULL");

	// Create visualizations for footprint and circles
	createObjects();
	mHaveFootprint = updateFootprints();

	mModelObserver.reset(new RigidModelObserver(*this, mModel));
	mModelObserver->registerCollisionChangedCallback(boost::bind(&RigidModelVisualization2D::collisionChanged, this));

	ok("Model");

	return true;
}

void RigidModelVisualization2D::clearObjects() {
	mModelObserver.reset();
	if(mInnerCircleItem) {
		mItem->removeFromGroup(mInnerCircleItem);
		delete mInnerCircleItem;
		mInnerCircleItem = NULL;
	}
	if(mOuterCircleItem) {
		mItem->removeFromGroup(mOuterCircleItem);
		delete mOuterCircleItem;
		mOuterCircleItem = NULL;
	}
	if(mFootprintItem) {
		mItem->removeFromGroup(mFootprintItem);
		delete mFootprintItem;
		mFootprintItem = NULL;
	}
	if(mXAxisItem) {
		mItem->removeFromGroup(mXAxisItem);
		delete mXAxisItem;
		mXAxisItem = NULL;
	}
}

void RigidModelVisualization2D::createObjects() {
	mItem->setPos(0, 0);
	mItem->setRotation(0);
	if(!mInnerCircleItem) {
		mInnerCircleItem = new QGraphicsEllipseItem();
		mItem->addToGroup(mInnerCircleItem);
	}
	if(!mOuterCircleItem) {
		mOuterCircleItem = new QGraphicsEllipseItem();
		mItem->addToGroup(mOuterCircleItem);
	}
	if(!mFootprintItem) {
		mFootprintItem = new QGraphicsPathItem();
		mItem->addToGroup(mFootprintItem);
	}
	if(!mXAxisItem) {
		mXAxisItem = new QGraphicsLineItem();
		mItem->addToGroup(mXAxisItem);
	}
	updateObjects();
}

bool RigidModelVisualization2D::updateFootprints() {
	if(!mModel || !mFootprintItem || !mOuterCircleItem || !mInnerCircleItem)
		return false;

	// Will throw if transform tree isn't available yet
	Footprint footprint;
	try {
		footprint = mModel->getFootprint("", Time::now(), "");
		if(footprint.empty())
			return false;
	} catch(...) {
		return false;
	}

	const float inner = footprint.getInnerRadius();
	const float outer = footprint.getOuterRadius();

	// Construct the footprint as a painter path consisting of n polygons
	QPainterPath path;
	foreach(auto poly, footprint) {
		QPolygonF fp_part;
		foreach(auto point, poly)
			fp_part.push_back(QPointF(point(0), point(1)));
		path.addPolygon(fp_part);
	}

	mFootprintItem->setPath(path);
	mInnerCircleItem->setRect(-inner, -inner, 2 * inner, 2 * inner);
	mOuterCircleItem->setRect(-outer, -outer, 2 * outer, 2 * outer);
	mXAxisItem->setLine(0.0, 0.0, inner, 0.0);

	// Toggle colors and visibility
	updateObjects();

	// Need the frame to visualize it later on
	try {
		mRootLinkFrame = mModel->getRootLink();
	} catch(...) {
		mRootLinkFrame = "";
		return false;
	}

	return true;
}

void RigidModelVisualization2D::collisionChanged() {
	if(!mModel)
		return;
	mHaveFootprint = updateFootprints();
}

void RigidModelVisualization2D::updateObjects() {
	const qreal zbase = 1.0; // let's lift ourselves above regular visualizations a bit

	if(mInnerCircleItem) {
		mInnerCircleItem->setPen(QPen(Qt::green, 0.0));
		mInnerCircleItem->setBrush(Qt::NoBrush);
		mInnerCircleItem->setZValue(zbase + 0.2);
		mInnerCircleItem->setVisible(mShowFootprintCircles);
	}
	if(mOuterCircleItem) {
		mOuterCircleItem->setPen(QPen(Qt::yellow, 0.0));
		mOuterCircleItem->setBrush(Qt::NoBrush);
		mOuterCircleItem->setZValue(zbase + 0.1);
		mOuterCircleItem->setVisible(mShowFootprintCircles);
	}
	if(mFootprintItem) {
		mFootprintItem->setPen(mFillFootprint ? Qt::NoPen : QPen(mFootprintColor, 0.0));
		mFootprintItem->setBrush(!mFillFootprint ? Qt::NoBrush : QBrush(mFootprintColor));
		mFootprintItem->setZValue(zbase + 0.0);
		mFootprintItem->setVisible(mShowFootprint);
	}
	if(mXAxisItem) {
		mXAxisItem->setPen(QPen(QColor(255, 200, 200), 0.0));
		mXAxisItem->setZValue(zbase + 0.3);
		mXAxisItem->setVisible(mShowXAxis);
	}
}

//////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::model::RigidModelVisualization2D, mira::Visualization2D);
