/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JointPublisherView.h
 *    View plugin for miracenter for modifying joints of rigid models.
 *
 * @author Tim Langner
 * @date   2011/12/08
 */

#ifndef _MIRA_JOINTPUBLISHERVIEW_H_
#define _MIRA_JOINTPUBLISHERVIEW_H_

#include <QTimer>
#include <QVBoxLayout>
#include <QLineEdit>
#include <QSlider>

#include <fw/Framework.h>
#include <rcp/ViewPart.h>
#include <views/PropertyViewPage.h>

#include <model/RigidModelProvider.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * View plugin for miracenter for modifying joints of rigid models.
 * Shows sliders and edit fields for all REVOLUTE, PRISMATIC and CONTINUOUS
 * joints of a rigid model that is obtained using either the IRigidModelProvider
 * service interface or the service provided in the RigidModelProvider parameter.
 */
class JointPublisherView : public ViewPart
{
MIRA_META_OBJECT(JointPublisherView,
				("Name", "JointPublisher")
				("Category", "Models")
				("Description", "Modify and publish joints of a rigid model"))
Q_OBJECT

public:
	JointPublisherView();
	virtual ~JointPublisherView();

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("RigidModelProvider", modelProvider, "Service used for providing a rigid model", "");
	}

	virtual Object* getAdapter(const Class& adapter);

protected:

	virtual QWidget* createPartControl();

public:

	std::string modelProvider;

private:
	PropertyViewPage* mControl;

	class UI;
	UI* ui;
};

//////////////////////////////////////////////////////////////////////////////

class JointPublisherView::UI : public QWidget
{
	Q_OBJECT
public:
	UI(JointPublisherView* parent);

public slots:

	void updateService();
	void revolute(int value);
	void prismatic(int value);
	void revolute(const QString& value);
	void prismatic(const QString& value);

public:

	void setupModel();
	void clearModel();

	QTimer* serviceTimer;
	Authority authority;
	std::string service;
	RigidModelPtr model;

private:
	struct JointInfo
	{
		float min;
		float max;
		QLineEdit* edit;
		QSlider* slider;
	};
	std::map<std::string, JointInfo> mJointInfos;
	JointPublisherView* mParent;
	QVBoxLayout* mLayout;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif 
