/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationGraph.h
 *    TODO Add description.
 *
 * @author Ronny Stricker
 * @date   2014/03/11
 */

#ifndef _MIRA_VISUALIZATIONGRAPH_H_
#define _MIRA_VISUALIZATIONGRAPH_H_

#include <geometry/Point.h>

#include <visualization/Visualization.h>
#include <views/VisualizationGraphView.h>

class QGraphicsItem;
class QGraphicsScene;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Base class for graph visualizations
template <typename T>
class VisualizationGraph :
	public Visualization
{
	MIRA_ABSTRACT_OBJECT(VisualizationGraph)

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	VisualizationGraph(std::string const channelDisplayName = "Channel") : Visualization(),
		mChannelDisplayName(channelDisplayName)
	{
		mDataChannel.setDataChangedCallback(
				boost::bind(&VisualizationGraph::dataChangedCallback, this, _1));
		mDataChannel.setChannelChangedCallback(
				boost::bind(&VisualizationGraph::channelChangedCallback, this));

	}

	/// The destructor.
	virtual ~VisualizationGraph()
	{
	}

	//@}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization);
		channelProperty(r, mChannelDisplayName.c_str(), mDataChannel, "The channel to be visualized");
	}

public:
	void init(IVisualizationSite* site)
	{
		mSite = dynamic_cast<IVisualizationGraphSite*>(site);
		assert(mSite!=NULL);
		setEnabled(isEnabled());
	}

	IVisualizationGraphSite* getSite()
	{
		return mSite;
	}


	virtual DataConnection getDataConnection() {
		return DataConnection(mDataChannel);
	}

protected:

	virtual void update(Duration dt) {
		if (!mDataChannel.isValid() || mDataChannel.getDataUpdateCount() == 0)
			return;
	}

	void dataChangedCallback(ChannelRead<T> data) {
		dataChanged(data);
	}

	void channelChangedCallback() {
		channelChanged();
	}

	/**
	 * Overwrite this method to get notified when the connected channel changes.
	 */
	virtual void channelChanged() {
		auto channel = mDataChannel.getChannel();
	}

protected:

	virtual void dataChanged(ChannelRead<T> data) = 0;

protected:
	ChannelProperty<T> mDataChannel;
	std::string mChannelDisplayName;

private:
	IVisualizationGraphSite* mSite;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
