#!/bin/bash

function usage
{
	echo -e "Usage: miramk <package>:<build config> [<package>:<build config> ...] [make options]"
	echo -e "       miramk <build config> [make options]\n"
	echo -e "Starts the build process of the specified packages using the given build configuration."
	echo -e "<build config> can be: debug, release, relwithdebinfo, etc.\n"
	echo -e "Example:"
	echo -e "       miramk MyPackage:release -j3\n"
	echo -e "If no package is given, ALL packages are built using the given configuration, e.g.:"
	echo -e "       miramk release -j3\n"
	exit 0
}

# enters the debug/release/test branch of the specified package within
# the corresponding build directory
# precondition: must be in directory of package
# parameter: $1:debug/release/test/...
function enterbuild
{
	local relpath updir subdir basedir
	relpath=""
	while true; do
		if [ -d "$PWD/build/" ];
		then
			basedir=$PWD
			if [ ! -f "build/$1/Makefile" ];
			then
				echo "The build/$1 directory does not exist yet, creating build environment now ..."
				make $basedir/build/$1/Makefile
			fi
			
			# go into target directory
			cd $basedir/build/$1/$relpath &> /dev/null
			if [ $? != 0 ]; then
				echo "The build directory 'build/$1/$relpath' does not exist yet, refreshing build environment now ..."
				cd $basedir/build/$1/ 
				make depend
				# try again to enter target directory
				cd $basedir/build/$1/$relpath &> /dev/null
				if [ $? != 0 ]; then
					echo "The build directory 'build/$1/$relpath' does not exist ..."
					return 255
				fi
			fi
			
			return 0
		fi
		
		updir=`dirname $PWD`
		subdir=`basename $PWD`
		
		if [[ "$updir" = "/" ]];
		then
			echo "There is no matching build/$1 directory for this package"
			return 255
		fi
		
		relpath="$subdir/$relpath"
		cd $updir
	done
}


function buildpackage #"$package" "$config" "${makeparams}"
{
	echo -e "\e[01;32mBuilding $1:$2 ...\e[00m"

	miracd $1
	if [ $? != 0 ]; then
		echo -e "Failed to enter build directory."
		return 1
	fi
	
	# check if CMakeLists exists
	if [ -f "CMakeLists.txt" ]; then

		enterbuild $2
		if [ $? != 0 ]; then
			return 1
		fi
	
		# start make
		make $3
		ret=$?
	else
		echo -e "No action required."
		ret=0
	fi
	return $ret
}

# we need miracd from mirabash, so import that file
DIR=$(dirname "${BASH_SOURCE[0]}")
source $DIR/../mirabash

if [[ $1 = "--help" ]] || [[ $1 = "-h" ]] || [[ $# -lt 1 ]]; then
	usage
fi

# the first parameters are the packages and the targets, the others are the make parameters, so
# find the first parameter, that does not contain a :colon, it will be the first parameter to 
# pass to make
paramcount=$#
firstmakeoption=$((${paramcount}+1))

for (( i=1; i<=${paramcount}; i++ ))
do
	if [[ ! ${!i} =~ .+:.* ]]; then
		firstmakeoption=$i
		break
	fi
done

if [[ $firstmakeoption == 1 ]]; then
	# first param is configuration (release/debug/...)
	config=$1
	firstmakeoption=$(($firstmakeoption+1)) #make options start one later
	makeparams="${@:${firstmakeoption}}"
	
	echo "Running 'make $config $makeparams' for all packages ..."
	echo "Reindexing packages ..." 
	$_mirafindpkg --reindex
	allpackages=($(bash $_mirafindpkg | awk -- '{print $1}'))
	num=${#allpackages[@]}
	
	failures=""
	for (( i=0; i<num; i++ ));
	do
		echo -e "\e[01;32m[$(($i+1))/$num] \e[00m\c"
		package="${allpackages[$i]}"
		
		if [[ $package == "external" ]]; then
			echo -e "\e[01;32mSkipping external ...\e[00m"
			continue;
		fi;
		
		buildpackage "$package" "$config" "${makeparams}"
		if [ $? != 0 ]; then
			echo -e "\e[01;31mFailure while building package '$package'.\e[00m"
			failures="$failures$package, "
			# continue
		fi
	done
	echo "Done."
	if [[ ! -z "$failures" ]]; then
		echo "Build failed for: $failures"
	fi;
	
else 
	makeparams="${@:${firstmakeoption}}"
	# build each given package
	for (( i=1; i<${firstmakeoption}; i++ ))
	do
		package="${!i%%:*}"
		config="${!i##*:}"
		buildpackage "$package" "$config" "${makeparams}"
		ret=$?
		if [ $ret != 0 ]; then
			echo -e "\e[01;31mFailure while building package '$package'. Abort!\e[00m"
			exit $ret
		fi
	done
	echo "Done."
fi

exit 0
