###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provides MIRA_ADD_JAR macro for invoking compiling Java sources into a JAR
# package
#
# Usage:
#     MIRA_ADD_JAR(TargetName
#                  [SOURCE     <the java source files>]
#                  [SELFSIGN]  for self-signing the jar file
#                  )
#
# Author: Erik Einhorn
#
###############################################################################

FIND_PACKAGE(Java 1.6)

SET(Java_KEYTOOL_EXECUTABLE keytool)
SET(Java_JARSIGNER_EXECUTABLE jarsigner)

INCLUDE(CMakeParseArguments)

MACRO(MIRA_ADD_JAR target)

	###########################################################################
	# Parse the macro arguments

	CMAKE_PARSE_ARGUMENTS(arg
		# options
		"SELFSIGN"
		# one-value arguments:
		"COMPONENT"
		# multi-value arguments:
		"SOURCE;LINK_LIBS;INSTALL_FILES"
		${ARGN})

	# Ensure, that we have at least one source file
	IF (NOT arg_SOURCE)
		MESSAGE(FATAL_ERROR "Target ${target} has no source files.")
	ENDIF()

	# Ensure, that INSTALL_FILES has a even number of elements
	SET(installFilesLen 0)
	LIST(LENGTH arg_INSTALL_FILES installFilesLen)
	SET(installFilesModulo 0)
	MATH(EXPR installFilesModulo ${installFilesLen}%2)
	IF (NOT installFilesModulo EQUAL 0)
		MESSAGE(FATAL_ERROR "Target ${target} has odd number of elements in INSTALL_FILES")
	ENDIF()

	###########################################################################
	# compile, create jar and install

	SET(CLASS_DEPS)	
	FOREACH(SOURCE_FILE ${arg_SOURCE})
		GET_FILENAME_COMPONENT(SOURCE_FILE_NAME ${SOURCE_FILE} NAME_WE)

		SET(CLASS_FILE_NAME "${SOURCE_FILE_NAME}.class")
		SET(OUTPUT_PATH "${CMAKE_CURRENT_BINARY_DIR}${CMAKE_FILES_DIRECTORY}/${target}.dir")

		ADD_CUSTOM_COMMAND(
			OUTPUT "${OUTPUT_PATH}/${CLASS_FILE_NAME}"
			COMMAND ${Java_JAVAC_EXECUTABLE}
				${CMAKE_JAVA_COMPILE_FLAGS}
				#-classpath ${JAVA_INCLUDE_PATH}
				d ${OUTPUT_PATH}
				${SOURCE_FILE}
			WORKING_DIRECTORY ${CMAKE_CURRENT_SOURCE_DIR}
			DEPENDS ${arg_SOURCE}
			COMMENT "Compiling ${SOURCE_FILE}"
		)
		LIST(APPEND CLASS_DEPS "${OUTPUT_PATH}/${CLASS_FILE_NAME}")
	ENDFOREACH(SOURCE_FILE)

	SET(JAR_OUTPUT_NAME "${target}.jar")
	SET(JAR_OUTPUT_PATH "${CMAKE_CURRENT_BINARY_DIR}/${JAR_OUTPUT_NAME}")

	ADD_CUSTOM_COMMAND(
		OUTPUT "${JAR_OUTPUT_PATH}"
		COMMAND ${Java_JAR_EXECUTABLE}
			-cf ${JAR_OUTPUT_PATH}
			"*.class"
		DEPENDS ${CLASS_DEPS}
		WORKING_DIRECTORY ${OUTPUT_PATH}
		COMMENT "Generating JAR archive ${JAR_OUTPUT_NAME}"
	)

	# for self signing:
	SET(KEYSTORE_FILE "${CMAKE_CURRENT_BINARY_DIR}/${target}.keystore")
	ADD_CUSTOM_COMMAND(
		OUTPUT "${KEYSTORE_FILE}"
		COMMAND ${Java_KEYTOOL_EXECUTABLE} -genkey -keystore ${KEYSTORE_FILE}
			-alias ${target} -keyalg RSA -storepass bigsecret
			-keypass bigsecret -validity 9999 -dname "cn=MIRA"
		COMMENT "Generating key for signing ${JAR_OUTPUT_NAME}"
	)

	ADD_CUSTOM_TARGET(${target}_signing
		COMMAND ${Java_JARSIGNER_EXECUTABLE} -storepass bigsecret
			-keystore ${KEYSTORE_FILE} ${JAR_OUTPUT_NAME} ${target}
		DEPENDS ${KEYSTORE_FILE} "${JAR_OUTPUT_PATH}"
		COMMENT "Signing ${JAR_OUTPUT_NAME}"
	)

	ADD_CUSTOM_TARGET(${target} ALL DEPENDS "${JAR_OUTPUT_PATH}")

	IF(arg_SELFSIGN)
		ADD_DEPENDENCIES(${target} ${target}_signing) 
	ENDIF(arg_SELFSIGN)

ENDMACRO(MIRA_ADD_JAR)
