###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provides MIRA_ADD_BINARY macro as alternative for the build-in
# ADD_EXECUTABLE macro. This macro perform the following steps:
#   1.) Add the binary as a new target using ADD_EXECUTABLE
#   2.) Handle Qt4 ui-Files, moc's, resources and translations
#   3.) Link the library using the build-it TARGET_LINK_LIBRARIES
#   4.) Install the binary to bin
#   5.) Install optional other files to somewhere
#
# Note for Qt translations: 
#   The name of the ts-files must be specified by using QT_TRs. The files to
#   translate must be specified by using QT_TR_SOURCES. The build chain will
#   created a separate qrc-file for the qm-files and compile this qrc-file into
#   the executable. To avoid conflicts with other resources, an optional prefix
#   for the translation can be specified by using QT_TR_PREFIX.
#
# Usage:
#    MIRA_ADD_BINARY(TargetName
#                   [PACKAGE        <package>]
#                   [COMPONENT      <component>]      <obsolete!>
#                   [SOURCE	        <source1 ... sourceN>]
#                   [QT_UIs         <ui1 ... uiN>]
#                   [QT_MOCs        <moc1 ... mocN>]
#                   [QT_QRCs        <qrc1 ... qrcN>]
#                   [QT_TRs         <file1 ... fileN>]
#                   [QT_TR_SOURCES  <file1 ... fileN>]
#                   [QT_TR_INCLUDES <dir1 ... dirN>]
#                   [QT_TR_PREFIX   <prefix>]
#                   [LINK_LIBS	    <lib1 ... libN>]
#                   [INSTALL_FILES  <src1 destDir1 ... srcN destDirN>]
#                   [DONT_INSTALL]
#                   )
#
# Author: Christian Martin
#
###############################################################################

INCLUDE(CMakeParseArguments)

###############################################################################

MACRO(MIRA_ADD_BINARY target)

	###########################################################################
	# Parse the macro arguments

	CMAKE_PARSE_ARGUMENTS(arg
		# options:
		"DONT_INSTALL"
		# one-value arguments:
		"PACKAGE;COMPONENT;INSTALL_PATH;QT_TR_PREFIX"
		# multi-value arguments:
		"SOURCE;QT_UIs;QT_MOCs;QT_QRCs;QT_TRs;QT_TR_SOURCES;QT_TR_INCLUDES;LINK_LIBS;INSTALL_FILES"
		${ARGN})

	###########################################################################

	IF (arg_PACKAGE)
		IF (arg_COMPONENT)
			MESSAGE(FATAL_ERROR "Target ${target}: PACKAGE and COMPONENT can't be mixed.")
		ENDIF()
		SET(arg_COMPONENT   ${arg_PACKAGE})
	ENDIF()

	# Ensure, that we have at least one source file
	IF (NOT arg_SOURCE)
		MESSAGE(FATAL_ERROR "Target ${target} has no source files.")
	ENDIF()

	# Ensure, that INSTALL_FILES has a even number of elements
	SET(installFilesLen 0)
	LIST(LENGTH arg_INSTALL_FILES installFilesLen)
	SET(installFilesModulo 0)
	MATH(EXPR installFilesModulo ${installFilesLen}%2)
	IF (NOT installFilesModulo EQUAL 0)
		MESSAGE(FATAL_ERROR "Target ${target} has odd number of elements in INSTALL_FILES")
	ENDIF()

	# Test if the obsolete INSTALL_PATH is given
	IF (arg_INSTALL_PATH)
		MESSAGE("-- WARNING - Target ${target}: "
		        "Obsolete option INSTALL_PATH is defined in MIRA_ADD_BINARY. "
		        "This option will be removed soon!")
	ENDIF()

	###########################################################################

	# Qt4 ui-Files
	SET(QT_UI_HDRS "")
	IF (arg_QT_UIs)
		QT_WRAP_UI(QT_UI_HDRS ${arg_QT_UIs})
		INCLUDE_DIRECTORIES(${CMAKE_CURRENT_BINARY_DIR})
	ENDIF()

	# Qt4 moc
	SET(QT_MOC_SRC "")
	IF (arg_QT_MOCs)
		# Avoid problems related to the fact that moc do not expand macro properly
		# by defining BOOST_TT_HAS_OPERATOR_HPP_INCLUDED. For some more details
		# please read here: https://bugreports.qt-project.org/browse/QTBUG-22829
		QT_WRAP_CPP(QT_MOC_SRC ${arg_QT_MOCs})
	ENDIF()

	# Qt4 resources
	SET(QT_RCC_SRCS "")
	IF (arg_QT_QRCs)
		QT_ADD_RESOURCES(QT_RCC_SRCS ${arg_QT_QRCs})
	ENDIF()

	# Qt translations
	SET(QT_QM_FILES "")
	IF (arg_QT_TRs)
		IF (NOT arg_QT_TR_SOURCES)
			MESSAGE(FATAL_ERROR "Target ${target}: Missing arguments for QT_TR_SOURCES. Need at least one file to translate.")
		ENDIF()
		MIRA_QT_CREATE_TRANSLATION(QT_QM_FILES QT_QM_QRC_FILE ${target}
				TRANSLATIONS ${arg_QT_TRs}
				FILES        ${arg_QT_TR_SOURCES}
				INCLUDES     ${arg_QT_TR_INCLUDES}
				OPTIONS      -locations none
				QRCPREFIX    ${arg_QT_TR_PREFIX}
				)
		# add the generated qrc file for the translations
		QT_ADD_RESOURCES(QT_RCC_SRCS ${QT_QM_QRC_FILE})
	ENDIF()

	###########################################################################
	# compile, link and install

	# add the target
	ADD_EXECUTABLE(${target} ${arg_SOURCE} ${QT_UI_HDRS} ${QT_MOC_SRC} ${QT_RCC_SRCS})

	get_target_property(COMPILE_FLAGS ${target} COMPILE_FLAGS)		
	if (${COMPILE_FLAGS} STREQUAL "COMPILE_FLAGS-NOTFOUND")
		set(COMPILE_FLAGS "")
	endif()

	# define PACKAGE and PACKAGE_VERSION identifiers to be used in code
	if (arg_PACKAGE)
		set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE=${arg_PACKAGE}")
		# MIRA_VERSION sets Package_Version based on major,minor,patch, so these 3 will also exist
		if ((DEFINED ${arg_PACKAGE}_MAJOR) AND (DEFINED ${arg_PACKAGE}_MINOR) AND (DEFINED ${arg_PACKAGE}_PATCH))
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION_MAJOR=${${arg_PACKAGE}_MAJOR}")
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION_MINOR=${${arg_PACKAGE}_MINOR}")
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION_PATCH=${${arg_PACKAGE}_PATCH}")
			math(EXPR VERSION_NUMBER "(${${arg_PACKAGE}_MAJOR}*1000+${${arg_PACKAGE}_MINOR})*100+${${arg_PACKAGE}_PATCH}")
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION=${VERSION_NUMBER}")
		else()
			message("-- WARNING - Package ${arg_PACKAGE}: Version unknown or incomplete. Please use MIRA_VERSION().")
		endif()			
	else()
		message("-- WARNING - Target ${target}: Cannot determine the package this target belongs to. Please set the PACKAGE parameter.")
	endif()

	set(COMPILE_FLAGS "${COMPILE_FLAGS} -DMIRA_PATHS=$ENV{MIRA_PATH}")

	set_target_properties(${target} PROPERTIES
 		COMPILE_FLAGS "${COMPILE_FLAGS}"
	)
	
	IF (MIRAAutoLinkLibraries)
		LIST(APPEND arg_LINK_LIBS ${MIRAAutoLinkLibraries})
	ENDIF()

	# add -rdynamic / --export-dynamic to the linker
	#   (This option is needed for some uses of "dlopen" or to allow obtaining
	#    backtraces from within a program.)
	SET_TARGET_PROPERTIES(${target} PROPERTIES ENABLE_EXPORTS 1)

	# link the binary
	TARGET_LINK_LIBRARIES(${target} ${arg_LINK_LIBS})

	IF (arg_PACKAGE)
		# Add binary to package target
		# check if target already exists
		IF( NOT TARGET PACKAGE_${arg_PACKAGE} )
			ADD_CUSTOM_TARGET( PACKAGE_${arg_PACKAGE} )
		ENDIF()
		# add dependency for package target
		ADD_DEPENDENCIES(PACKAGE_${arg_PACKAGE} ${target})
		# set label property (for CDash)
		SET_PROPERTY(TARGET ${target} PROPERTY LABELS ${arg_PACKAGE})
	ENDIF()

	# install the binary to bin only if NOT_INSTALL is not specified
	if (NOT arg_DONT_INSTALL)
		MIRA_DIST_BINARY(${target})
	endif()

	# install all optional files
	SET(installFilesIdx 0)
	WHILE(installFilesIdx LESS installFilesLen)
		SET(fileSrc "")
		SET(fileDest "")
		LIST(GET arg_INSTALL_FILES ${installFilesIdx} fileSrc)
		MATH(EXPR installFilesIdx ${installFilesIdx}+1)
		LIST(GET arg_INSTALL_FILES ${installFilesIdx} fileDest)
		MATH(EXPR installFilesIdx ${installFilesIdx}+1)

		MIRA_DIST_FILE(${target} ${fileSrc} ${fileDest})
	ENDWHILE()

ENDMACRO(MIRA_ADD_BINARY)
