/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file QImageTest.C
 *    Test conversion mira::Img -> QImage
 *
 * @author Christof Schröter
 * @date   2022/09/16
 */

#include <boost/test/unit_test.hpp>

#include <math/Random.h>
#include <math/Saturate.h>

#include <serialization/adapters/boost/optional.hpp>

#include <widgets/QtUtils.h>

using namespace mira;

namespace singlechan {

///////////////////////////////////////////////////////////////////////////////

template <typename T>
void randomFill(Img<T, 1>& img)
{
	for (int y = 0; y < img.height(); ++y)
		for (int x = 0; x < img.width(); ++x) {
			if (std::is_integral<T>::value) // runtime-check is not elegant but will do here
				img(x, y) = MIRA_RANDOM.uniform((int)std::numeric_limits<T>::lowest(),
				                                (int)std::numeric_limits<T>::max()+1);
			else
				img(x, y) = MIRA_RANDOM.uniform(std::numeric_limits<T>::lowest(),
				                                std::numeric_limits<T>::max());
		}
}

// reproduce transform as (should be) implemented in makeQImage, for reference

template <typename T>
uint8 transform(T val)
{
	return val;
}

// specialization int8, add 128
template <>
uint8 transform(int8 val)
{
	return val + 128;
}

template <typename T>
uint8 transform(T val, T minVal, T maxVal)
{
	return saturate<T>(255 * (val - minVal) / (maxVal-minVal), 0, 255);
}

// specialization uint16, calc in int
template <>
uint8 transform(uint16 val, uint16 minVal, uint16 maxVal)
{
	return saturate<int>(255 * ((int)val - (int)minVal) / (int)(maxVal-minVal), 0, 255);
}

// specialization int16, calc in int
template <>
uint8 transform(int16 val, int16 minVal, int16 maxVal)
{
	return saturate<int>(255 * ((int)val - (int)minVal) / (int)(maxVal-minVal), 0, 255);
}

template <typename T>
void compare(const Img<T, 1>& img, const QImage& qimg,
             boost::optional<T> minVal = boost::optional<T>(),
             boost::optional<T> maxVal = boost::optional<T>())
{
	for (int y = 0; y < img.height(); ++y)
		for (int x = 0; x < img.width(); ++x) {
			if (maxVal) {
				BOOST_CHECK_EQUAL(transform(img(x,y), *minVal, *maxVal), qBlue(qimg.pixel(x,y)));
				BOOST_CHECK_EQUAL(transform(img(x,y), *minVal, *maxVal), qGreen(qimg.pixel(x,y)));
				BOOST_CHECK_EQUAL(transform(img(x,y), *minVal, *maxVal), qRed(qimg.pixel(x,y)));
			} else {
				BOOST_CHECK_EQUAL(transform(img(x,y)), qBlue(qimg.pixel(x,y)));
				BOOST_CHECK_EQUAL(transform(img(x,y)), qGreen(qimg.pixel(x,y)));
				BOOST_CHECK_EQUAL(transform(img(x,y)), qRed(qimg.pixel(x,y)));
			}
			BOOST_CHECK_EQUAL(255, qAlpha(qimg.pixel(x,y)));
		}
}

template <typename T>
void testMakeQImage(boost::optional<T> minVal = boost::optional<T>(),
                    boost::optional<T> maxVal = boost::optional<T>())
{
	Img<T, 1> img(101, 101); // odd numbers chosen deliberately
	randomFill(img);

	QImage qimg;
	if (maxVal) {
		// must be called with scaling params!
		BOOST_CHECK_THROW(QtUtils::makeQImage(img), mira::XInvalidParameter);

		qimg = QtUtils::makeQImage(img, *minVal, *maxVal);
	} else {
		// cannot be called with scaling params!
		BOOST_CHECK_THROW(QtUtils::makeQImage(img, 0, 0), mira::XInvalidParameter);

		qimg = QtUtils::makeQImage(img);
	}

	compare(img, qimg, minVal, maxVal);

	Img<T, 1> img2 = img.clone();
	QImage qimg2;
	if (maxVal)
		qimg2 = QtUtils::makeQImage(img2, *minVal, *maxVal);
	else
		qimg2 = QtUtils::makeQImage(img2);

	// change img2, should not affect qimg2
	randomFill(img2);
	BOOST_CHECK(qimg == qimg2);

	QImage qimg3;
	// call through interface for generic Img<>, should give same result
	if (maxVal)
		qimg3 = QtUtils::makeQImage((Img<>)img, *minVal, *maxVal);
	else
		qimg3 = QtUtils::makeQImage((Img<>)img);
	BOOST_CHECK(qimg == qimg3);
}

template <typename T>
void testMakeQImage(T minVal, T maxVal)
{
	// older gcc wants explicit initialization for boost::optional, this method does just that
	testMakeQImage(boost::optional<T>(minVal), boost::optional<T>(maxVal));
}

///////////////////////////////////////////////////////////////////////////////

} // namespace singlechan

namespace multichan {

///////////////////////////////////////////////////////////////////////////////

template <int C>
void randomFill(Img<uint8, C>& img)
{
	for (int y = 0; y < img.height(); ++y)
		for (int x = 0; x < img.width(); ++x)
			for (int c = 0; c < C; ++c)
				img(x, y)[c] = MIRA_RANDOM.uniform<int>(0, 256);
}

template <int C>
void compare(const Img<uint8, C>& img, const QImage& qimg)
{
	for (int y = 0; y < img.height(); ++y) {
		for (int x = 0; x < img.width(); ++x) {
			BOOST_CHECK_EQUAL(img(x,y)[0], qBlue(qimg.pixel(x,y)));
			BOOST_CHECK_EQUAL(img(x,y)[1], qGreen(qimg.pixel(x,y)));
			BOOST_CHECK_EQUAL(img(x,y)[2], qRed(qimg.pixel(x,y)));
			if (C == 3)
				BOOST_CHECK_EQUAL(255, qAlpha(qimg.pixel(x,y)));
			else
				BOOST_CHECK_EQUAL(img(x,y)[3], qAlpha(qimg.pixel(x,y)));
		}
	}
}

template <typename ImgT>
void testMakeQImage()
{
	ImgT img(101, 101); // odd numbers chosen deliberately
	randomFill(img);

	// cannot be called with scaling params!
	BOOST_CHECK_THROW(QtUtils::makeQImage(img, 0, 0), mira::XInvalidParameter);

	QImage qimg = QtUtils::makeQImage(img);
	compare(img, qimg);

	ImgT img2 = img.clone();
	QImage qimg2 = QtUtils::makeQImage(img2); // qimg2 is a clone of qimg

	// change img2, should not affect qimg2
	randomFill(img2);
	BOOST_CHECK(qimg == qimg2);

	// call through interface for generic Img<>, should give same result
	QImage qimg3 = QtUtils::makeQImage((Img<>)img);
	BOOST_CHECK(qimg == qimg3);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace multichan {

BOOST_AUTO_TEST_CASE(makeQImage )
{
	singlechan::testMakeQImage<uint8>();
	singlechan::testMakeQImage<int8>();

	singlechan::testMakeQImage<uint16>(0, 100);
	singlechan::testMakeQImage<uint16>(50, 1000);
	singlechan::testMakeQImage<int16>(-1000, 1000);

	singlechan::testMakeQImage<float>(-1.23e4, 1234.f);
	singlechan::testMakeQImage<double>(1234., 12345.);

	multichan::testMakeQImage<Img8U3>();
	multichan::testMakeQImage<Img8U4>();
}

BOOST_AUTO_TEST_CASE( toQImageUnsupportedFormat )
{
	Img32F2 img(1, 1);
	BOOST_CHECK_THROW(QtUtils::makeQImage(img), mira::XNotImplemented);
}

BOOST_AUTO_TEST_CASE( asQImageImg8U1 )
{
	Img8U1 img(100, 101); // width must be multiple of 4 (odd width is tested separately below)
	singlechan::randomFill(img);

	QImage qimg = QtUtils::asQImage(img);
	singlechan::compare(img, qimg);
	BOOST_CHECK_EQUAL(img.data(0), qimg.scanLine(0));

	// change img, qimg should change accordingly
	singlechan::randomFill(img);
	singlechan::compare(img, qimg);

	const Img8U1& cref = img;
	qimg = QtUtils::asQImage(cref);
	singlechan::compare(img, qimg);

	// memory is NOT shared when provided through CONST Img8U1&!
	const QImage& qimgref = qimg; // avoid triggering a deep-copy ourselves here
	for (int y = 0; y < img.height(); ++y)
		BOOST_CHECK_NE(img.data(y), qimgref.scanLine(y));

	// now, changing img should not affect qimg
	QImage qimg2 = qimg.copy();
	singlechan::randomFill(img);
	BOOST_CHECK(qimg == qimg2);
}

BOOST_AUTO_TEST_CASE( asQImageImg8U4 )
{
	Img8U4 img(101, 101); // odd numbers chosen deliberately
	multichan::randomFill(img);

	QImage qimg = QtUtils::asQImage(img);
	multichan::compare(img, qimg);
	BOOST_CHECK_EQUAL(img.data(0), qimg.scanLine(0));

	// change img, qimg should change accordingly
	multichan::randomFill(img);
	multichan::compare(img, qimg);

	const Img8U4& cref = img;
	qimg = QtUtils::asQImage(cref);
	multichan::compare(img, qimg);

	const QImage& qimgref = qimg; // qimg.scanLine(0) would trigger a deep-copy, 
	                              // as original data is read-only when provided through const&
	BOOST_CHECK_EQUAL(img.data(0), qimgref.scanLine(0));

	// change img (again), qimg should change accordingly
	multichan::randomFill(img);
	multichan::compare(img, qimg);

	// non-const access to qimg (like qimg.scanLine()) will trigger deep-copy -> stop sharing
	for (int y = 0; y < img.height(); ++y)
		BOOST_CHECK_NE(img.data(y), qimg.scanLine(y));

	// image content should still be the same (completely copied)
	multichan::compare(img, qimg);

	// now, changing img should not affect qimg
	QImage qimg2 = qimg.copy();
	multichan::randomFill(img);
	BOOST_CHECK(qimg == qimg2);
}

BOOST_AUTO_TEST_CASE( asQImageUnsupportedFormat )
{
	Img8U1 img(101, 101);
	BOOST_CHECK_THROW(QtUtils::asQImage(img), mira::XInvalidParameter);
}

