/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SpinBoxDelegate.C
 *    Implementation of SpinBoxDelegate.h
 *
 * @author Christian Reuther
 * @date   2012/02/07
 */

#include <widgets/SpinBoxDelegate.h>

#include <QDoubleSpinBox>
#include <math/Truncate.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

QDoubleSpinBox* createDoubleSpinBox(QWidget* parent, double minRange,
                                    double maxRange, double step,
                                    double defvalue, const QString& suffix,
                                    bool buttons, bool readonly,
                                    unsigned int decimals) {
	QDoubleSpinBox* box = new QDoubleSpinBox(parent);

	box->setDecimals(decimals);
	box->setRange(minRange, maxRange);
	box->setValue(defvalue);
	box->setSingleStep(step);
	box->setSuffix(suffix);
	box->setLocale(QLocale(QLocale::English));
	box->setReadOnly(readonly);
	box->setButtonSymbols(buttons ? QAbstractSpinBox::UpDownArrows : QAbstractSpinBox::NoButtons);

	return box;
}

QSpinBox* createSpinBox(QWidget* parent, int min, int max, int step,
                        int defvalue, const QString& suffix, bool buttons, bool readonly) {
	QSpinBox* box = new QSpinBox(parent);

	box->setRange(min, max);
	box->setValue(defvalue);
	box->setSingleStep(step);
	box->setSuffix(suffix);
	box->setLocale(QLocale(QLocale::English));
	box->setReadOnly(readonly);
	box->setButtonSymbols(buttons ? QAbstractSpinBox::UpDownArrows : QAbstractSpinBox::NoButtons);

	return box;
}

////////////////////////////////////////////////////////////////////////////////

SpinBoxDelegate::SpinBoxDelegate(QObject *parent) : QStyledItemDelegate(parent) {
	// Set some default values
	setRange(-9999.99, 9999.99, 0.1);
	setDefaultValue(0.0);
	setSuffix("");
	setShowButtons(true);
	setDecimals(2);
}

SpinBoxDelegate::~SpinBoxDelegate() {
	// Empty
}

////////////////////////////////////////////////////////////////////////////////

void SpinBoxDelegate::setSpinBoxFormat(double minRange, double maxRange, double step,
                                       double defvalue, const QString& suffix,
                                       bool buttons, unsigned int decimals) {
	setRange(minRange, maxRange, step);
	setDefaultValue(defvalue);
	setSuffix(suffix);
	setShowButtons(buttons);
	setDecimals(decimals);
}

void SpinBoxDelegate::setRange(double minRange, double maxRange, double step) {
	mMinRange = minRange;
	mMaxRange = maxRange;
	mStep = step;
}

void SpinBoxDelegate::setDefaultValue(double defvalue) {
	mDefValue = defvalue;
}

void SpinBoxDelegate::setSuffix(const QString& suffix) {
	mSuffix = suffix;
}

void SpinBoxDelegate::setShowButtons(bool show) {
	mShowButtons = show;
}

void SpinBoxDelegate::setDecimals(unsigned int decimals) {
	mDecimals = decimals;
}

////////////////////////////////////////////////////////////////////////////////

QWidget *SpinBoxDelegate::createEditor(QWidget *parent,
                                       const QStyleOptionViewItem &/* option */,
                                       const QModelIndex &/* index */) const {
	QDoubleSpinBox *editor = createDoubleSpinBox(parent, mMinRange, mMaxRange,
		mStep, mDefValue, mSuffix, mShowButtons, false, mDecimals);

	return editor;
}

void SpinBoxDelegate::setEditorData(QWidget *editor, const QModelIndex &index) const {
	const double value = index.model()->data(index, Qt::EditRole).toDouble();

	QDoubleSpinBox *spinBox = static_cast<QDoubleSpinBox*>(editor);
	spinBox->setValue(value);
}

void SpinBoxDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                   const QModelIndex &index) const {
	QDoubleSpinBox *spinBox = static_cast<QDoubleSpinBox*> (editor);
	spinBox->interpretText();

	model->setData(index, spinBox->value(), Qt::EditRole);
}

void SpinBoxDelegate::updateEditorGeometry(QWidget *editor,
                                           const QStyleOptionViewItem &option,
                                           const QModelIndex &/* index */) const {
	editor->setGeometry(option.rect);
}

QString SpinBoxDelegate::displayText(const QVariant &value,
                                     const QLocale &locale) const {
	return locale.toString(value.toDouble(), 'f', mDecimals) + mSuffix;
}

////////////////////////////////////////////////////////////////////////////////

}
