/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CollapsibleTreeHeader.C
 *    Implementation of CollapsibleTreeHeader.h.
 *
 * @author Ronny Stricker
 * @date   21.08.2013
 */

#include <widgets/CollapsibleTreeHeader.h>

#include <QTreeView>
#include <QMouseEvent>
#include <QPainter>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

CollapsibleTreeHeader::CollapsibleTreeHeader( Qt::Orientation orientation,
                                              QTreeView* parent) :
			QHeaderView(orientation, parent),
			state(QIcon::Off),
			button( new QPushButton )
{
	QIcon lockIcon;
	lockIcon.addPixmap(QPixmap(":/icons/Collapse.png"), QIcon::Normal,
		QIcon::Off);
	lockIcon.addPixmap(QPixmap(":/icons/Expand.png"), QIcon::Normal,
		QIcon::On);

	button->setIcon(lockIcon);
	button->setIconSize( QPixmap(":/icons/Expand.png").size() );
	setStretchLastSection( true );
}

void CollapsibleTreeHeader::paintSection( QPainter *painter,
                                          QRect const& rect,
                                          int logicalIndex) const
{
	painter->save();
	// call paint section with the reduced space for the first column
	// otherwise the header label will not be centered in the remaining header
	// area which looks a bit odd
	if (logicalIndex != 0)
		QHeaderView::paintSection(painter, rect, logicalIndex);
	else
		QHeaderView::paintSection(painter,
			QRect(rect.left() + height(), rect.top(),
				rect.width() - height(), rect.height()), logicalIndex);
	painter->restore();

	// we want to have a collapse/expand button in the first column
	if (logicalIndex == 0) {

		// we need a background panel first of all
		QStyleOptionFrame frameOption;
		frameOption.rect =  QRect(0,0, height(), height() );
		frameOption.state = QStyle::State_None;

		style()->drawPrimitive( QStyle::PE_FrameDockWidget, &frameOption, painter );

		// add a pushbutton
		QStyleOptionButton option;
		option.rect = QRect(0, 0, height()-1, height()-1);
		option.state =
				button->isDown() ? QStyle::State_On : QStyle::State_Off;

		style()->drawControl(QStyle::CE_PushButton, &option, painter,
			button.data());

		// and finally we need to add the pixmap
		int pixmapSize = button->iconSize().height() > height()-4 ?
				height()-4 : button->iconSize().height();

		painter->drawPixmap(
			QPoint( (height() - pixmapSize) / 2, (height() - pixmapSize) / 2),
			button->icon().pixmap( QSize( pixmapSize, pixmapSize )
				, QIcon::Normal, state ) );
	}
}

void CollapsibleTreeHeader::mousePressEvent(QMouseEvent *event)
{
	// change state of button to "Down" if the button is pressed
	if (QRect(0, 0, height(), height()).contains(event->pos())
			&& !button->isDown()) {
		button->setDown(true);
		viewport()->update();
	}
	QHeaderView::mousePressEvent(event);
}

void CollapsibleTreeHeader::mouseReleaseEvent(QMouseEvent *event)
{
	// switch state of button back to normal if mouse is released
	if (button->isDown()) {
		button->setDown(false);
		// if the mouse is released on the button (click event) we have to
		// collapse or expand the tree widget
		if (QRect(0, 0, height(), height()).contains(event->pos())) {
			state = state == QIcon::On ? QIcon::Off : QIcon::On;
			foldTree();
		}
		viewport()->update();
	}

	QHeaderView::mouseReleaseEvent(event);
}

void CollapsibleTreeHeader::foldTree()
{
	QTreeView* widget = dynamic_cast<QTreeView*>( parent() );
	if ( widget ) {
		if ( state == QIcon::On )
			widget->collapseAll();
		else
			widget->expandAll();
	}
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

