/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeasureTool.C
 *    A tool for the 3D visualization to measure distance and angle between two points or along
 *    a path.
 *
 * @author Tim Langner, Christof Schröter
 * @date   2014/07/17
 */


#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreSceneManager.h>
#include <QMouseEvent>
#include <QToolTip>

#include <transform/Pose.h>

#include <visualization/3d/SceneQueryTool.h>
#include <visualization/3d/ArrowObject.h>
#include <visualization/3d/LineStripObject.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Tool for selecting two points and measuring angle and distance between both in the yx-plane of a 3D scene
 * (in a Visualization3DView)
 */
class MeasureTool : public SceneQueryTool
{
	MIRA_META_OBJECT(MeasureTool,
			("Name", "Measure")
			("Category", "Interact")
			("Description", "Measure distance and angle")
			("Default", "false")
			("Order", "400"))
	enum SelectionMode
	{
		First,
		Second,
	};

public:

	MeasureTool();

public:
	/** @name Implementation of VisualizationTool */
	//@{

	virtual void activate();

	virtual void onMousePressed(QMouseEvent* e);

	virtual void onMouseMoved(QMouseEvent* e);

	//@}

	virtual QIcon getIcon();

protected:
	/** @name Implementation of VisualizationTool3D */
	//@{

	virtual void setupScene(IVisualization3DSite* site);

	//@}

private:

	std::pair<std::string, Ogre::Vector3> getIntersection(float x, float y,
	                                                      const std::string& mask="");

private:

	boost::shared_ptr<ArrowObject> mArrowObject;
	SelectionMode mSelectionMode;
	std::string mSelectedObject;
	bool mIsValid;

	Point2f mFirstPoint;

	boost::shared_ptr<LineStripObject> mLineStripObject;
	float mLineLength;
};

///////////////////////////////////////////////////////////////////////////////

MeasureTool::MeasureTool() :
	mIsValid(false)
{}

QIcon MeasureTool::getIcon()
{
	QPixmap pixmap(":/icons/ToolGoal.png");
	return QIcon(pixmap);
}

void MeasureTool::activate()
{
	VisualizationTool3D::activate();

	if (mIsValid)
		mArrowObject->setVisible(true);
	else
		mArrowObject->setVisible(false);
}

void MeasureTool::onMousePressed(QMouseEvent* e)
{
	// The right mouse button disables/deactivates the tool
	if (e->button() == Qt::RightButton ||
			(e->button() == Qt::LeftButton && mSelectionMode == Second && e->modifiers() != Qt::ShiftModifier))
	{
		mIsValid = false;
		mSelectedObject = "";
		mSelectionMode = First;
		QToolTip::hideText();
		mArrowObject->setVisible(false);

		mLineStripObject->clear();
		mLineLength = 0;

		return;
	}

	if (e->button() != Qt::LeftButton)
		return;

	auto i = getIntersection(e->pos().x(), e->pos().y(), mSelectedObject);
	if (!i.first.empty())
	{
		// shift+click accumulates path
		if (mSelectionMode == Second && e->modifiers() == Qt::ShiftModifier)
		{
			mLineStripObject->setVisible(true);

			Point2f secondPoint(i.second.x, i.second.y);
			Point2f diff = secondPoint-mFirstPoint;
			mLineLength += diff.norm();
			// no return here, just treat this as the first click for the next segment
		}

		mLineStripObject->add(i.second);

		mArrowObject->setPosition(i.second);
		mArrowObject->setLength(0);
		mSelectedObject = i.first;
		mFirstPoint.x() = i.second.x;
		mFirstPoint.y() = i.second.y;
		mSelectionMode = Second;
		mIsValid = true;
		mArrowObject->setVisible(true);
	}
}

void MeasureTool::onMouseMoved(QMouseEvent* e)
{
	if (!mSelectedObject.empty())
	{
		auto i = getIntersection(e->pos().x(), e->pos().y(), mSelectedObject);
		if (i.first.empty())
			return;

		Point2f secondPoint(i.second.x, i.second.y);
		Point2f diff = secondPoint-mFirstPoint;
		float length = diff.norm();

		switch (mSelectionMode) {
			case Second:
			{
				float phi = std::atan2(diff.y(), diff.x());
				mArrowObject->setOrientation(Eigen::Quaternionf(Eigen::AngleAxisf(phi-half_pi<float>(), Eigen::Vector3f::UnitZ())));
				mArrowObject->setLength(length);
				QString text = QString("From: (X=%1 Y=%2) To: (X=%3 Y=%4) Angle=%5deg Length=%6m")
									.arg(mFirstPoint.x(), 0, 'f', 3)
									.arg(mFirstPoint.y(), 0, 'f', 3)
									.arg(secondPoint.x(), 0, 'f', 3)
									.arg(secondPoint.y(), 0, 'f', 3)
									.arg(SignedAnglef(phi).deg(), 0, 'f', 2)
									.arg(length, 0, 'f', 3);
				if (mLineLength > 0)
					text += QString(" , Path=%6m").arg(length+mLineLength, 0, 'f', 3);

				text += "\n\nClick to finish, Shift+Click to add path segments";
				QToolTip::showText(e->globalPos(), text);
				break;
			}
			default: break;
		}
	}
}

void MeasureTool::setupScene(IVisualization3DSite* site)
{
	SceneQueryTool::setupScene(site);
	mArrowObject.reset(new ArrowObject(site->getSceneManager(), mNode));
	mArrowObject->setRadius(0.025f);
	mArrowObject->setVisible(false);
	mSelectedObject = "";
	mSelectionMode = First;

	mLineStripObject.reset(new LineStripObject(site->getSceneManager(), mNode));
	mLineStripObject->setLineWidth(0);
	mLineStripObject->setColor(Ogre::ColourValue::Red);
	mLineStripObject->setVisible(false);
	mLineLength = 0;
}

std::pair<std::string, Ogre::Vector3> MeasureTool::getIntersection(float x, float y,
                                                                   const std::string& mask)
{
	return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject", false, true)
	                    : SceneQueryTool::getIntersection(x, y, mask, true, true);
}


///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::MeasureTool, mira::VisualizationTool3D)
