/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeFileDialog.C
 *    Implementation of TapeFileDialog.h.
 *
 * @author Tim Langner
 * @date   2011/12/30
 */

#include <fw/TapeFileDialog.h>

#include <QGridLayout>
#include <QHBoxLayout>
#include <QSplitter>
#include <QLabel>

#include <fw/Tape.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapeFileDialog::TapeFileDialog(QWidget* parent) :
	QFileDialog(parent),
	mPreview(NULL), mUseOriginalTimestamp(NULL), mNamespacePrefix(NULL)
{
	// Force to use the Qt-Layout. Otherwise, the layout() might be NULL and
	// no preview can be added.
	setOption(QFileDialog::DontUseNativeDialog, true);

	QGridLayout* gridLayout = dynamic_cast<QGridLayout*>(this->layout());
	if (gridLayout != NULL)
	{
		QLayoutItem* layoutItem = gridLayout->itemAtPosition(1,0);
		QSplitter* splitter;
		splitter = dynamic_cast<QSplitter*>(layoutItem->widget());

		if (splitter != NULL)
		{
			QFrame* frame = new QFrame(splitter);
			splitter->addWidget(frame);

			QVBoxLayout* layout = new QVBoxLayout(frame);
			layout->setContentsMargins(0, 0, 0, 0);

			mPreview = new QTextEdit(frame);
			mPreview->setReadOnly(true);
			mPreview->setText("No file selected");

			mUseOriginalTimestamp = new QCheckBox("Use original start time for playback", frame);

			QHBoxLayout* hLayout = new QHBoxLayout();
			QLabel* nsLabel = new QLabel("Namespace prefix");
			mNamespacePrefix = new QLineEdit("");
			hLayout->addWidget(nsLabel);
			hLayout->addWidget(mNamespacePrefix);

			layout->addWidget(mPreview);
			layout->addWidget(mUseOriginalTimestamp);
			layout->addLayout(hLayout);

			frame->setLayout(layout);

			connect(this, SIGNAL(currentChanged(QString)), this, SLOT(setPreview(QString)));
		}
	}
}

void TapeFileDialog::setPreview(const QString &file)
{
	try
	{
		Tape t;
		t.open(file.toLocal8Bit().data(), Tape::INFO);
		std::stringstream s;
		Tape::ChannelMap channels = t.getChannels();

		uint32 v = t.getVersion();
		uint16 major = (uint16)(v >> 16);
		uint16 minor = (uint16)(v & 0x00FF);

		s << "Tape file:\n"
			<< "  version : " << major << "." << minor << "\n"
			<< "  recorded: " << t.getLocalStartTime() << ")\n"
			<< "  channels: " << channels.size() << "\n\n";

		foreach(auto c, channels)
			s << " Channel " << c.first << "\n"
				<< "   type    : " << c.second.type << "\n"
				<< "   messages: " << c.second.messages.size() << "\n"
				<< "   first   : " << c.second.firstMessageOffset << "\n"
				<< "   last    : " << c.second.lastMessageOffset << "\n\n";
		mPreview->setText(s.str().c_str());
	}
	catch(...)
	{
		mPreview->setText("This is not a tape file");
	}
}

///////////////////////////////////////////////////////////////////////////////

}

