/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OgreWidgetEx.h
 *    Extended more advanced derived class of OgreWidget
 *
 * @author Erik Einhorn
 * @date   2012/09/01
 */

#ifndef _MIRA_OGREWIDGETEX_H_
#define _MIRA_OGREWIDGETEX_H_

#ifndef Q_MOC_RUN
#include <OGRE/OgreColourValue.h>
#endif

#include <geometry/Point.h>
#include <widgets/OgreWidget.h>


namespace Ogre {
	class Camera;
	class Light;
	class Viewport;
	class SceneManager;
}

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Extended version of OgreWidget that provides default setup of Ogre
 * components (scene manager, light, camera, etc.) and hence provides a
 * ready-to-use playground for 3D rendering.
 *
 * It moreover provides advanced capabilities such as:
 * - different background modes
 * - 3D stereo modes
 * - wireframe rendering modes
 * - different camera modes (perspective, orthographic)
 */
class OgreWidgetEx : public OgreWidget
{
public:
	enum CameraMode
	{
		PERSPECTIVE=0,
		ORTHOGRAPHIC,
		BIRDS_EYE
	};

	enum StereoMode
	{
		NONE,
		CROSSED_EYE
	};

	enum BackgroundMode
	{
		BACKGROUND_SOLID=0,
		BACKGROUND_GRADIENT,
		BACKGROUND_WHITE,
		BACKGROUND_BLACK
	};

	enum RenderMode
	{
		NORMAL = 0,
		WIREFRAME,
		HIDDEN_LINE
	};

public:

	explicit OgreWidgetEx(QWidget* parent=0);
	virtual ~OgreWidgetEx();

protected:

	virtual void initOgre();

public:

	void setBackgroundMode(BackgroundMode mode);
	void setBackgroundColor1(Ogre::ColourValue color1);
	void setBackgroundColor2(Ogre::ColourValue color2);
	void setBackgroundColors(Ogre::ColourValue color1, Ogre::ColourValue color2);

	void setCameraMode(CameraMode mode);
	void setRenderMode(RenderMode mode);
	void setStereoMode(StereoMode mode);

	void showBoundingBoxes(bool show=true);


	void setCamera(const Point3f& position, float yaw, float pitch, float roll, float distance);
	void lookAt(const Point3f& from, const Point3f& to);

protected:

	void updateCamera();


protected:

	class ColouredRectangle2D;
	class RenderModeTechniqueMod;

	// background style
	BackgroundMode mBackgroundMode;
	Ogre::ColourValue mBackgroundColor1;
	Ogre::ColourValue mBackgroundColor2;

	// camera mode
	CameraMode mCameraMode;
	struct {
		Point3f position;
		float yaw, pitch, roll;
		float distance;
	} mCameraSettings;

	// render mode
	RenderMode mRenderMode;

	// stereo mode
	StereoMode mStereoMode;
	float mStereoEyeDistance;

	bool mShowBoundingBoxes;

	// components: light, camera, etc
	Ogre::Light* mLight;
	Ogre::Camera* mCamera; // main / left camera
	Ogre::Camera* mCameraRight; // right camera
	Ogre::Viewport* mViewport; // main / left viewport
	Ogre::Viewport* mViewportRight; // right view port
	Ogre::SceneManager* mSceneManager;
	ColouredRectangle2D* mBackground;
	RenderModeTechniqueMod* mRenderModeTechniqueMod;

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
