/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OgreUtils.h
 *    Declaration of helper functions for converting from and to Ogre.
 *
 * @author Erik Einhorn
 * @date   2010/12/19
 */

#ifndef _MIRA_OGREUTILS_H_
#define _MIRA_OGREUTILS_H_

#include <geometry/Point.h>
#include <utils/Path.h>
#include <image/Color.h>

#include <transform/Pose.h>

#ifndef Q_MOC_RUN
#include <OGRE/OgreString.h>
#include <OGRE/OgreVector3.h>
#include <OGRE/OgreColourValue.h>
#include <OGRE/OgreSceneQuery.h>
#endif

///////////////////////////////////////////////////////////////////////////////

namespace Ogre {
	class SceneManager;
	class SceneNode;
	class ManualObject;
	class MovableObject;
}

///////////////////////////////////////////////////////////////////////////////

namespace mira { namespace OgreUtils {

///////////////////////////////////////////////////////////////////////////////

template <typename TPoint>
inline Ogre::Vector3 toOgreVector(const TPoint& p)
{
	static_assert(sizeof(TPoint)==0, "toOgreVector is not implemented for this type");
	return Ogre::Vector3();
}

template <>
inline Ogre::Vector3 toOgreVector<Point2f>(const Point2f& p)
{
	return Ogre::Vector3(p.x(), p.y(), 0);
}

template <>
inline Ogre::Vector3 toOgreVector<Point3f>(const Point3f& p)
{
	return Ogre::Vector3(p.x(), p.y(), p.z());
}

template <>
inline Ogre::Vector3 toOgreVector<Pose2>(const Pose2& p)
{
	return Ogre::Vector3(p.x(), p.y(), 0);
}

template <>
inline Ogre::Vector3 toOgreVector<PoseCov2>(const PoseCov2& p)
{
	return Ogre::Vector3(p.x(), p.y(), 0);
}

template <>
inline Ogre::Vector3 toOgreVector<Pose3>(const Pose3& p)
{
	return Ogre::Vector3(p.x(), p.y(), p.z());
}

template <>
inline Ogre::Vector3 toOgreVector<PoseCov3>(const PoseCov3& p)
{
	return Ogre::Vector3(p.x(), p.y(), p.z());
}

template <>
inline Ogre::Vector3 toOgreVector<RigidTransform<double, 2> >(const RigidTransform<double, 2>& p)
{
	return Ogre::Vector3(p.x(), p.y(), 0);
}

template <>
inline Ogre::Vector3 toOgreVector<RigidTransformCov<double, 2> >(const RigidTransformCov<double, 2>& p)
{
	return Ogre::Vector3(p.x(), p.y(), 0);
}

template <>
inline Ogre::Vector3 toOgreVector<RigidTransform<double, 3> >(const RigidTransform<double, 3>& p)
{
	return Ogre::Vector3(p.x(), p.y(), p.z());
}

template <>
inline Ogre::Vector3 toOgreVector<RigidTransformCov<double, 3> >(const RigidTransformCov<double, 3>& p)
{
	return Ogre::Vector3(p.x(), p.y(), p.z());
}

template <typename ColorType>
inline Ogre::ColourValue toOgreColor(const ColorType& c)
{
	return Ogre::ColourValue();
}

template <>
inline Ogre::ColourValue toOgreColor<Color::RGB>(const Color::RGB& c)
{
	return Ogre::ColourValue(c.r, c.g, c.b, 1.0f);
}

template <>
inline Ogre::ColourValue toOgreColor<Color::RGBA>(const Color::RGBA& c)
{
	return Ogre::ColourValue(c.r, c.g, c.b, c.a);
}

///////////////////////////////////////////////////////////////////////////////

void loadResource(const Path& path,
                                          const Ogre::String& resourceGroup);

/**
 * Sets position and orientation of the scene node according to the
 * specified transform.
 */
void setTransform(Ogre::SceneNode* node,
                                          const RigidTransform2f& t);

/**
 * Sets position and orientation of the scene node according to the
 * specified transform.
 */
void setTransform(Ogre::SceneNode* node,
                                          const RigidTransform3f& t);

/**
 * Sets the position of the  scene node according to the
 * specified translation.
 */
void setPosition(Ogre::SceneNode* node,
                                         const Eigen::Vector2f& t);

/**
 * Sets the position of the  scene node according to the
 * specified translation.
 */
void setPosition(Ogre::SceneNode* node,
                                         const Eigen::Vector3f& t);

/**
 * Sets the orientation of the scene node according to the
 * specified rotation.
 */
void setOrientation(Ogre::SceneNode* node,
                                            const Eigen::Rotation2D<float>& r);

/**
 * Sets the orientation of the scene node according to the
 * specified rotation.
 */
void setOrientation(Ogre::SceneNode* node,
                                            const Eigen::Quaternion<float>& r);

/**
 * Returns true if object is directly or indirectly attached to the given node.
 */
bool hasAttachedObject(Ogre::SceneNode* node,
                                               Ogre::MovableObject* object);


/**
 * Performs ray intersection tests for ALL movable objects in the specified
 * scene node and all of its child nodes.
 * All movable objects that are hit by the ray are stored in oResult.
 */
void rayQuery(Ogre::SceneNode* node,
                                      const Ogre::Ray& ray,
                                      Ogre::RaySceneQueryResult& oResult,
                                      uint32 queryMask=0xFFFFFFFF);


// intersection point: ray.getPoint(result.second);
/**
 * Performs a ray/object collision check.
 * If the ray collides with the object, this method returns
 * (true, distance to collision). If the ray does not collide with the
 * object the method returns (false, ?).
 * The intersection point can be computed as: ray.getPoint(result.second);
 * For an Entity and ManualObject this method performs the tests down to
 * polygon level.
 */
std::pair<bool,float> rayObjectCollision(const Ogre::Ray& ray,
                                             const Ogre::MovableObject* object);

///////////////////////////////////////////////////////////////////////////////

}}

#endif
