/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LineStripObject.h
 *    Renders a line strip as billboard chain. In contrast to line
 *    strips of manual objects, this class supports lines with a width
 *    above 1 pixel.
 *
 * @author Erik Einhorn
 * @date   2012/11/06
 */

#ifndef _MIRA_LINESTRIPOBJECT_H_
#define _MIRA_LINESTRIPOBJECT_H_

#ifndef Q_MOC_RUN
#include <OGRE/OgreMaterial.h>
#endif

#include <visualization/3d/VisualizationObject.h>

///////////////////////////////////////////////////////////////////////////////

namespace Ogre {
	class BillboardChain;
}

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Renders one or more line strips. This class supports lines whose width is
 * larger than one pixel (in contrast to Ogre's build-in line strip support).
 * If the line width is set to 0 or negative, a one pixel wide line is rendered.
 * A new line strip can be added by calling the begin() method, followed by
 * several calls to point() in order to add points to the strip. Finally, the
 * strip must be finished by calling the end() method.
 * All line strips can be cleared using the clear() method.
 * The class also provides support for line lists, which can be produced by
 * creating multiple line strips consisting of two points only, or by
 * using the addLine() methods.
 */
class LineStripObject : public VisualizationObject
{
public:
	LineStripObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent = NULL);
	virtual ~LineStripObject();

	/// Resets the color of all elements of the line (overwriting all possible individual colors of the line points)
	virtual void setColor(const Ogre::ColourValue& color);

	virtual void setVisible(bool visible, bool cascade = true);
	bool isVisible() const { return mIsVisible; }

	/// Enables/disables alpha blending.
	void enableAlphaBlending(bool useAlpha=true);

	/**
	 * Sets the width of the line.
	 * If width <= 0 a one pixel wide line will be shown.
	 */
	void setLineWidth(float width);

public:

	/// Clears all line strips and points.
	void clear();

	/// Starts a new line strip.
	void begin();

	/// Finishes the current line strip.
	void end();

	/// Adds another point to the current line strip. This must be called between a call to begin() and end().
	void point(const Ogre::Vector3& point);

	/// Adds another point with the specified color to the current line strip. This must be called between a call to begin() and end().
	void point(const Ogre::Vector3& point, const Ogre::ColourValue& color);


	/**
	 * Adds another point to the last line strip. This method can be called
	 * outside of begin() and end(), but will be much slower than the
	 * begin()-point()...-end() procedure described above.
	 */
	void add(const Ogre::Vector3& point);

	/**
	 * Adds another point to the last line strip. This method can be called
	 * outside of begin() and end(), but will be much slower than the
	 * begin()-point()...-end() procedure described above.
	 */
	void add(const Ogre::Vector3& point, const Ogre::ColourValue& color);

	/// Adds a line (a line strip consiting of two points)
	void addLine(const Ogre::Vector3& p1,const Ogre::Vector3& p2);

	/// Adds a line with the specified color
	void addLine(const Ogre::Vector3& p1,const Ogre::Vector3& p2, const Ogre::ColourValue& color);

private:

	struct Point
	{
		Point() {}
		Point(const Ogre::Vector3& iP, const Ogre::ColourValue& iC) :
			p(iP), c(iC) {}

		Ogre::Vector3 p;
		Ogre::ColourValue c;
	};

	struct Strip
	{
		Strip() : pixelLineSection(-1) {}
		std::list<Point> pts;
		int pixelLineSection;
	};

private:

	void regenerateStrip(std::size_t idx);
	void regenerateAll();
	void manageVisibility();

private:

	std::vector<boost::shared_ptr<Strip>> mStrips;
	boost::shared_ptr<Strip> mActiveStrip;

	Ogre::BillboardChain* mBillboardLines; // for width >0
	Ogre::ManualObject* mPixelLines; // for width <=0

	Ogre::MaterialPtr mMaterial;
	float mWidth;
	Ogre::ColourValue mColor;

	bool mIsVisible;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
