/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Point3Visualization.C
 *    Example for a advanced visualization of a point by a sphere mesh.
 *
 * @author Tim Langner
 * @date   2011/06/25
 */

#include <visualization/Visualization3D.h>
#include <visualization/3d/MeshObject.h>
#include <OGRE/OgreSceneManager.h>
#include <widgets/OgreUtils.h>

using namespace mira;

class Point3Visualization :  public Visualization3D
{
	MIRA_META_OBJECT(Point3Visualization,
		("Name", "Point")
		("Category", "Geomtery")
		("Description", "Visualizes a point."))
public:
	Point3Visualization() :
		mPoint(NULL),
		mNode(NULL)
	{
		// register our callback function
		mPointChannel.setDataChangedCallback(boost::bind(&Point3Visualization::dataChanged, this, _1));
	}

	virtual ~Point3Visualization()
	{
		delete mPoint;
		if(getSite()==NULL)
			return;

		getSite()->getSceneManager()->destroySceneNode(mNode);
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		// call base class reflect
		MIRA_REFLECT_BASE(r, Visualization3D);
		// our channel property for a channel named "Point"
		channelProperty(r, "Point", mPointChannel, "The point channel to visualize");
	}

	virtual void setupScene(IVisualization3DSite* site)
	{
		Ogre::SceneManager* sceneManager = site->getSceneManager();
		mNode = sceneManager->getRootSceneNode()->createChildSceneNode();
		mPoint = new MeshObject("Sphere.mesh", sceneManager, mNode);
	}

	void dataChanged(ChannelRead<Point3f> read)
	{
		// set position of our scene point
		Ogre::Vector3 p = OgreUtils::toOgreVector(read->value());
		mPoint->setPosition(p);
		// store the frame id and timestamp of our data
		mFrameID = read->frameID;
		mDataTimestamp = read->timestamp;
	}

	virtual DataConnection getDataConnection()
	{
		return DataConnection(mPointChannel);
	}

protected:

	virtual void update(Duration dt)
	{
		if (!mPointChannel.isValid())
			return;
		// get the transformation between
		// our data frame and the selected frame of the 3D editor view
		RigidTransform3f d = getAuthority().getTransform<RigidTransform3f>(
			mFrameID, mDataTimestamp,
			getSite()->getCameraFrame(), Time::now(),
			getSite()->getFixedFrame());
		// set the transform on the node (indirectly move mPoint)
		OgreUtils::setTransform(mNode, d);
	}

protected:

	ChannelProperty<Point3f> mPointChannel;
	MeshObject* mPoint;
	Ogre::SceneNode* mNode;
	std::string mFrameID;
	Time mDataTimestamp;
};

MIRA_CLASS_SERIALIZATION(Point3Visualization, mira::Visualization3D);
