/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Point2Visualization.C
 *    Example for a 2d visualization of a point
 *
 * @author Tim Langner
 * @date   2011/06/25
 */

#include <visualization/Visualization2D.h>

#include <QGraphicsEllipseItem>
#include <QGraphicsScene>

#include <widgets/QtUtils.h>

using namespace mira;

class Point2Visualization :  public Visualization2D
{
	MIRA_META_OBJECT(Point2Visualization,
			("Name", "Point")
			("Category", "Geometry")
			("Description", "Visualizes a point."))
public:
	Point2Visualization() :
		mRoot(NULL),
		mEllipse(NULL)
	{
		mPointChannel.setDataChangedCallback(boost::bind(&Point2Visualization::dataChanged, this, _1));
	}

	virtual ~Point2Visualization()
	{
		delete mEllipse;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		// call base class reflect
		MIRA_REFLECT_BASE(r, Visualization2D);
		// our channel property for a channel named "Point"
		channelProperty(r, "Point", mPointChannel, "The point channel to visualize");
	}

	virtual void setupScene(IVisualization2DSite* site)
	{
		QGraphicsScene* sceneManager = site->getSceneManager();
		// we will use an ellipse to render our point
		mEllipse = new QGraphicsEllipseItem();
		// give it a size of 1 cm
		mEllipse->setRect(-0.05, -0.05, 0.1, 0.1);
		sceneManager->addItem(mEllipse);
	}

	virtual DataConnection getDataConnection()
	{
		return DataConnection(mPointChannel);
	}

	void dataChanged(ChannelRead<Point2f> read)
	{
		// set position of our scene point
		mEllipse->setPos(read->x(), read->y());
		mFrameID = read->frameID;
		mDataTimestamp = read->timestamp;
	}

protected:

	virtual void update(Duration dt)
	{
		if (!mPointChannel.isValid())
			return;
		// get the transformation between
		// our data frame and the selected frame of the 2D editor view
		RigidTransform2f d = getAuthority().getTransform<RigidTransform2f>(
			mFrameID, mDataTimestamp,
			getSite()->getCameraFrame(), Time::now(),
			getSite()->getFixedFrame());
		// set the transform on the node (indirectly move mEllipse)
		QtUtils::setTransform(mRoot, d);
	}

protected:

	ChannelProperty<Point2f> mPointChannel;
	QGraphicsItem* mRoot;
	QGraphicsEllipseItem* mEllipse;
	std::string mFrameID;
	Time mDataTimestamp;
};

MIRA_CLASS_SERIALIZATION(Point2Visualization, mira::Visualization2D);

