/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UnitManager.C
 *    Implementation of UnitManager.h.
 *
 * @author Tim Langner
 * @date   2011/02/25
 */

#include <fw/UnitManager.h>

#include <error/LoggingAux.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

UnitManager::~UnitManager()
{
	// stop all units
	stopUnits(true);

	// stop thread groups explicitly so that all finalize handlers
	// of grouped units will be called before they get destroyed in below
	// for loop
	foreach(auto p, mThreadGroups)
		p.second->stop();

	// destroy all units explicitly here in reverse order:
	foreach_reverse(UnitEntry& p, mUnits)
	{
		try
		{
			std::get<0>(p).reset();
		}
		catch(std::exception& ex)
		{
			MIRA_LOG_EXCEPTION(ERROR, ex) << "An exception has occurred while "
				"destructing the unit '" << std::get<2>(p) << "'." 
				<< "This indicates a problem within the unit. "
				"(Note: that destructors must NOT throw exceptions)\n";
		}
	}
}

std::string UnitManager::addThreadGroup(const ResourceName& ns,
                                        const std::string& name)
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	if(!ns.isFullyQualified())
		MIRA_THROW(XInvalidParameter, "Namespace '" << ns << "' is not fully qualified");
	std::string threadGroupID = ns / name;
	if (mThreadGroups.count(threadGroupID) != 0)
		return threadGroupID;
	boost::shared_ptr<Authority> tg(new Authority());
	tg->checkin(ns, name);
	mThreadGroups[tg->getGlobalID()] = tg;
	return tg->getGlobalID();
}

void UnitManager::addUnit(MicroUnitPtr unit, const std::string& ns,
                          const std::string& name, const std::string& threadGroup,
                          bool startAtInit)
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	mUnits.push_back(UnitEntry(unit, ns, name, threadGroup, startAtInit));
}

void UnitManager::removeUnit(const std::string& id)
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	auto it = mUnits.begin();
	for (; it != mUnits.end(); ++it)
	{
		ResourceName ns = std::get<1>(*it);
		std::string name = std::get<2>(*it);
		if (id == (ns / name))
			break;
	}
	if (it != mUnits.end())
		mUnits.erase(it);
}

MicroUnitPtr UnitManager::getUnit(const std::string& id)
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	foreach(const UnitEntry& p, mUnits)
	{
		MicroUnitPtr u = std::get<0>(p);
		ResourceName ns = std::get<1>(p);
		std::string name = std::get<2>(p);
		if (id == (ns / name))
			return u;
	}
	return MicroUnitPtr();
}

void UnitManager::initializeUnits()
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	foreach(UnitEntry& p, mUnits)
	{
		MicroUnitPtr u = std::get<0>(p);
		const std::string& ns = std::get<1>(p);
		std::string& id = std::get<2>(p); // this one is not const, we modify it below
		const std::string& threadGroup = std::get<3>(p);
		bool startAtInit = std::get<4>(p);

		if(!u->isValid())
		{
			// initialize and start the unit
			if (!threadGroup.empty())
			{
				auto tg = mThreadGroups.find(threadGroup);
				if (tg == mThreadGroups.end())
					MIRA_THROW(XInvalidConfig, "Unit has unknown thread group '"
					           << threadGroup << "'");
				u->checkin(*tg->second.get(), ns, id);
			}
			else
				u->checkin(ns,id);

			if(!u->isStarted() && startAtInit)
				u->start();
		}

		// make sure we store the id, NOT the name (only possible after checkin)
		id = u->getID();
	}

	foreach(auto p, mThreadGroups)
	{
		if(!p.second->isStarted())
			p.second->start();
	}
}

void UnitManager::startUnits()
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	foreach(const UnitEntry& p, mUnits)
	{
		MicroUnitPtr unit = std::get<0>(p);
		if(unit->isValid())
			unit->start();
	}
}

void UnitManager::stopUnits(bool destruct)
{
	boost::mutex::scoped_lock lock(mUnitsMutex);
	foreach_reverse(const UnitEntry& p, mUnits)
	{
		MicroUnitPtr unit = std::get<0>(p);
		if(unit->isValid())
		{
			if (destruct)
				unit->destruct();
			else
				unit->stop();
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
