/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeVisitor.C
 *    Implementation of TapeVisitor.h.
 *
 * @author Tim Langner
 * @date   2010/12/29
 */

#include <fw/TapeVisitor.h>
#include <serialization/BinaryJSONConverter.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void TapeVisitor::iterator::readJSON(Stamped<JSONValue>& oData)
{
	validate();

	// instantiate the data
	const MessageInstance& msgInst = this->operator*();
	Buffer<uint8> buf(msgInst.data);

	if ( !(mMessage->info->meta) )
		MIRA_THROW(XIO, "Meta information missing for channel \"" + mMessage->info->name + "\"!");

	// convert data to JSON
	BinaryJSONConverter::binaryToJSON( buf, mMessage->tape->getVersion() < 0x00050000,
		*mMessage->info->meta, mMessage->info->metaDB, oData.value() );

	// and fill in the header info
	oData.timestamp = mVisitor->getStartTime() + getTimeOffset();
	oData.frameID = msgInst.frameID;
	oData.sequenceID = msgInst.sequenceID;
}

void TapeVisitor::visit(Tape* tape, Duration first, Duration last, bool ignoreStartTime)
{
	uint32 messages = mMessages.size();

	const Tape::ChannelMap& channels = tape->getChannels();
	Tape::ChannelMap::const_iterator i = channels.begin();
	for( ; i != channels.end(); ++i)
		addChannel(tape, &i->second, first, last);

	// adapt time and reIndex only if we have added some messages
	if (mMessages.size() != messages)
		reIndex(tape, ignoreStartTime);
}

void TapeVisitor::visit(Tape* tape, const std::vector<std::string>& channels,
                        Duration first, Duration last, bool ignoreStartTime)
{
	uint32 messages = mMessages.size();
	for (uint32 i=0; i<channels.size(); ++i)
	{
		Tape::ChannelMap::const_iterator channel = tape->getChannels().find(channels[i]);
		if ( channel == tape->getChannels().end() )
			continue;
		addChannel(tape, &channel->second, first, last);
	}
	// adapt time and reIndex only if we have added some messages
	if (mMessages.size() != messages)
		reIndex(tape, ignoreStartTime);
}

void TapeVisitor::addChannel(Tape* tape, const Tape::ChannelInfo* channel,
                             Duration first, Duration last)
{
	ChannelMap::iterator c = mChannels.find(channel->name);
	if (c != mChannels.end())
	{
		if (c->second->type != channel->type)
			MIRA_THROW(XIO, "Channel " << channel->name
			           << " is already known to tape visitor but with different "
			           << "type " << c->second->type << " != " << channel->type);
	}
	else
		mChannels.insert(ChannelMap::value_type(channel->name, channel));

	for (uint32 i=0; i<channel->messages.size(); ++i)
	{
		if (channel->messages[i].timeOffset < first ||
			channel->messages[i].timeOffset > last)
			continue;
		MessageInfo message;
		message.tape = tape;
		message.info = channel;
		message.index = &channel->messages[i];
		message.timeOffset = channel->messages[i].timeOffset;
		mMessages.push_back(message);
	}
}

void TapeVisitor::reIndex(Tape* tape, bool ignoreStartTime)
{
	if (!mMessages.empty()) {
		const Duration& tz1 = mMessages.front().tape->getTZOffset();
		const Duration& tz2 = tape->getTZOffset();
		if (tz1 != tz2) {
			MIRA_LOG(WARNING) << "TapeVisitor: Different timezone offsets in visited tapes: "
			                  << tz1 << " / " << tz2;
		}
	}

	if (!ignoreStartTime && (!mStart || tape->getStartTime() < mStart))
		mStart = tape->getStartTime();
	if (mStart) {
		for (uint32 i=0; i<mMessages.size(); ++i)
			mMessages[i].timeOffset = (mMessages[i].tape->getStartTime() +
			                           mMessages[i].index->timeOffset) - *mStart;
	}
	std::sort(mMessages.begin(), mMessages.end());
}

///////////////////////////////////////////////////////////////////////////////

}
