/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Tape.C
 *    Implementation of Tape.h.
 *
 * @author Tim Langner
 * @date   2010/12/28
 */

#include <boost/filesystem.hpp>

#include <fw/Tape.h>
#include <fw/FrameworkDefines.h>

#include <utils/CompressBuffer.h>
#include <utils/PrintProgressBar.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

const uint32 Tape::sHeaderSize = sizeof(char)+sizeof(uint32);
const uint32 Tape::sMessageBlockSize = 2*sizeof(int64)+2*sizeof(uint32);

///////////////////////////////////////////////////////////////////////////////

void readChannelInfoEx(Buffer<uint8>& buffer, uint32 version,
                       uint64& oOffset, Tape::ChannelInfo& ioInfo)
{
	BinaryBufferIstream is(&buffer);
	int64 firstMessageOffset;
	int64 lastMessageOffset;
	ioInfo.dataSize = 0;
	if (version < 0x00020000)
	{
		uint32 dOffset, dIndex;
		is >> dOffset >> firstMessageOffset >> lastMessageOffset
			>> ioInfo.name >> ioInfo.type >> dIndex;
		oOffset = dOffset;
		ioInfo.offsetToIndexTable = dIndex;
	}
	else
	{
		is >> oOffset >> firstMessageOffset >> lastMessageOffset
			>> ioInfo.name >> ioInfo.type;
		if (version > 0x00020000)
		{
			Buffer<uint8> metaBuffer;
			is >> metaBuffer;
			if (version >= 0x00050000)
			{
				BinaryBufferDeserializer ds(&metaBuffer);
				ds.deserialize(ioInfo.meta);
				ds.deserialize(ioInfo.metaDB);
			}
			if (version >= 0x00060000)
				is >> ioInfo.dataSize;
		}
		is >> ioInfo.offsetToIndexTable;
	}
	if (version < 0x00010001)
	{
		firstMessageOffset *= 1000;
		lastMessageOffset *= 1000;
	}
	ioInfo.firstMessageOffset = Duration::nanoseconds(firstMessageOffset);
	ioInfo.lastMessageOffset = Duration::nanoseconds(lastMessageOffset);
}

Tape::~Tape()
{
	close();
}

uint32 Tape::getCurrentVersion()
{
	return MIRA_TAPE_VERSION;
}

std::size_t Tape::write(const std::string& channelID, const std::string& typeName,
                        const Time& time, const std::string& frameID, uint32 sequenceID,
                        const Buffer<uint8>& data, int compressionLevel,
                        TypeMetaPtr meta,
                        const MetaTypeDatabase& metaDB)
{
	// check open mode
	if (!mIsOpen || mMode != WRITE )
		MIRA_THROW(XIO, "Tape is not opened in write mode.");

	boost::mutex::scoped_lock lock(mMessageMutex);

	// the offset of the message time since start of recording
	Duration timeSinceStart = time - mFile.start;

	ChannelMap::iterator i = mChannels.find(channelID);
	// create channel info if channel was not recorded before
	if (i == mChannels.end())
	{
		i = mChannels.insert(ChannelMap::value_type(channelID, ChannelInfo())).first;
		i->second.name = channelID.c_str();
		i->second.type = typeName;
		i->second.meta = meta;
		i->second.metaDB = metaDB;
		i->second.dataSize = 0;
	}

	// store the message in the sort window message map
	MessageMap::iterator it = mMessages.insert(std::make_pair(timeSinceStart, Message()));
	it->second.name = channelID.c_str();
	it->second.frameID = frameID.c_str();
	it->second.sequenceID = sequenceID;
	if (compressionLevel != 0)
	{
		it->second.compressed = true;
		it->second.data.reserve(getMaxCompressedLength(data.size()));
		compressBuffer(data, it->second.data, compressionLevel);
	}
	else {
		it->second.compressed = false;
		it->second.data = data;
	}

	it->second.uncompressedSize = data.size();

	if (!mWaitForAlteredStartTime)
	{
		while (mMessages.size() > 0 &&
			  (mMessages.rbegin()->first - mMessages.begin()->first) > mSortWindowSize)
		{
			write(mMessages.begin()->first, mMessages.begin()->second);
			mMessages.erase(mMessages.begin());
		}
	}

	return it->second.data.size();
}


void Tape::write(Duration time, const Message& message)
{
	if (!mIsOpen || mMode != WRITE )
		MIRA_THROW(XIO, "Tape is not opened in write mode.");

	boost::mutex::scoped_lock lock(mMutex);

	// create a message block if we don't have one
	if (!mCurrentMessageBlock)
		startMessageBlock(time);
	auto i = mChannels.find(message.name);
	assert(i != mChannels.end());
	// write channel info if channel was not recorded before
	if (mWrittenChannels.count(message.name) == 0)
	{
		i->second.firstMessageOffset = time;
		writeChannelInfo(i->second);
		mWrittenChannels.insert(message.name);
	}

	mFile.file.seekp(0, std::ios::end);

	// create an index struct
	MessageIndex index;
	index.offset = (uint64)mFile.file.tellp()-mCurrentMessageBlock->offset;
	index.timeOffset = time;
	index.block = mCurrentMessageBlock->offset;

	// update the channel info
	// this message is the latest message
	i->second.lastMessageOffset = time;
	// add message to index vector
	i->second.messages.push_back(index);
	// add size of message
	i->second.dataSize += message.data.size();

	// serialize message with header and data
	Buffer<uint8> buffer;
	BinaryBufferOstream os(&buffer);
	os << time.totalNanoseconds() << message.name << message.frameID << message.sequenceID;

	// write compressed (if compressed or normal size if uncompressed) size,
	// compression flag and if compressed the uncompressed size
	os << (uint32)message.data.size() << message.compressed;
	if (message.compressed)
		os << message.uncompressedSize;

	writeHeader(MESSAGE, buffer.size());
	mFile.file.write((char*)buffer.data(), buffer.size());
	mFile.file.write((char*)message.data.data(), message.data.size());
	mFile.file.flush();
	// increase size of message block by size of data
	mCurrentMessageBlock->size += Tape::sHeaderSize+buffer.size()+message.data.size();
	// increase message count for active message block
	++mCurrentMessageBlock->nrMessages;

	// update message block time limits
	if (time < mCurrentMessageBlock->firstMessageOffset)
		mCurrentMessageBlock->firstMessageOffset = time;
	if (mCurrentMessageBlock->lastMessageOffset < time)
		mCurrentMessageBlock->lastMessageOffset = time;

	// if we exceed maximum message block size close this message block
	// and create a new one next time
	if (mCurrentMessageBlock->size > mMaxMessageBlockSize)
		finishMessageBlock();
}

void Tape::open(const Path& file, OpenMode mode)
{
	close();
	mChannels.clear();
	mMessageBlocks.clear();
	mMessages.clear();
	mCurrentMessageBlock = boost::optional<MessageBlock>();

	if ((mode != WRITE) && !boost::filesystem::is_regular_file(file))
		MIRA_THROW(XFileNotFound, file.string() + " does not exist or is not a file");

	mMode = mode;
	switch (mMode)
	{
		case INFO : openInfo(file); break;
		case READ : openRead(file); break;
		case WRITE : openWrite(file); break;
		default : return;
	}
	mIsOpen = true;
}

void Tape::close()
{
	if (mIsOpen && mMode == READ )
		closeRead();
	if (mIsOpen && mMode == WRITE )
		closeWrite();
	mFile.file.close();
	mIsOpen = false;
}

void Tape::repair(const Path& file, const Path& outFile, bool displayProgress)
{
	close();
	Path resolvedIn = resolvePath(file);
	if ( !boost::filesystem::exists(resolvedIn) )
		MIRA_THROW(XFileNotFound, "Tape file '" << resolvedIn.string()
		           << "' could not be found!");
	Path resolvedOut = resolvePath(outFile);
	if (resolvedIn == resolvedOut)
		MIRA_THROW(XIO, "Output tape file name must not equal input tape file");

	static const int32 SIZE_SCALE = 50000; // avoid int overflow (when calling printProgressBar())
	                                       // up to file size of INT_MAX*50000 = 100TB
	uint64 scaledFileSize;
	if (displayProgress)
	{
		std::ifstream tmpStream(resolvedIn.string().c_str(), std::ios::binary | std::ios::ate); // at end
		scaledFileSize = tmpStream.tellg() / SIZE_SCALE;
	}

	mFile.file.open(resolvedIn.string().c_str(), std::ios::in | std::ios::binary);
	if(!mFile.file.is_open())
		MIRA_THROW(XIO,"Failed to open tape '" << file.string() << "' for reading")

	readFileHeader();
	Tape outtape;
	outtape.open(outFile, Tape::WRITE);
	outtape.setStartTime(mFile.start);

	Time repairStart = Time::now();
	if (displayProgress)
		printProgressBarHeader();

	while(!mFile.file.eof())
	{
		uint64 offset = (uint64)mFile.file.tellp();
		if (displayProgress)
			printProgressBar(offset/SIZE_SCALE, scaledFileSize, repairStart);
		Header h = readHeader();
		// we found an invalid header so stop repairing
		if (h.type != CHANNELINFO && h.type != MESSAGEBLOCK && h.type != MESSAGE && h.type != INDEX)
			break;
		Buffer<uint8> buffer(h.size);
		mFile.file.read((char*)buffer.data(), h.size);
		// check if we failed to read the chunk and stop repairing
		if (mFile.file.fail())
			break;

		if (h.type == CHANNELINFO) {
			ChannelInfo info;
			readChannelInfoEx(buffer, mFile.version, offset, info);
			outtape.mChannels.insert(ChannelMap::value_type(info.name, info));
		}

		if (h.type == MESSAGE)
		{
			BinaryBufferIstream is(&buffer);
			int64 timeOffset;
			uint32 compressedMessageSize;
			Message message;

			is >> timeOffset >> message.name;
			if (mFile.version >= 0x00050000)
				is >> message.frameID >> message.sequenceID;
			is >> compressedMessageSize >> message.compressed;
			if (mFile.version < 0x00010001)
				timeOffset *= 1000;
			if (message.compressed)
				is >> message.uncompressedSize;

			auto it = outtape.mChannels.find(message.name);
			if(it==outtape.mChannels.end())
			{
				MIRA_LOG(WARNING) << "Failed to repair message from channel '" <<
				                     message.name << "' since no channel information is available";
			}
			Buffer<uint8> dataBuffer(compressedMessageSize);

			mFile.file.read((char*)dataBuffer.data(), compressedMessageSize);
			// check if we failed to read message and skip writing it to avoid
			// writing broken or unfinished messages
			if (mFile.file.fail())
				break;
			// if compressed -> uncompress first
			if (message.compressed)
			{
				message.data.reserve(message.uncompressedSize);
				uncompressBuffer(dataBuffer, message.data);
			}
			else
				message.data = std::move(dataBuffer);
			if (mFile.version < 0x00050000)
			{
				// extract stamped header from data and throw it away
				BinaryBufferIstream dataStream(&message.data);
				std::string tn;
				uint64 ts;
				dataStream >> tn >> ts >> message.frameID >> message.sequenceID;
				std::size_t headerSize = sizeof(uint32)+tn.size()				// size of serialized typename
										 + sizeof(uint64)						// size of serialized timestamp
										 + sizeof(uint32)+message.frameID.size()// size of serialized frame id
										 + sizeof(uint32);						// size of serialized sequence id
				message.data.pop_front(headerSize);
			}
			// message was compressed -> compress it again
			if (message.compressed)
			{
				Buffer<uint8> compressedData(getMaxCompressedLength(message.data.size()));
				compressBuffer(message.data, compressedData);
				message.data = std::move(compressedData);
			}
			outtape.write(Duration::nanoseconds(timeOffset), message);
		}
	}
	if (displayProgress)
		printFinishedProgressBar();
}

void Tape::waitForAlteredStartTime()
{
	// check open mode
	if (!mIsOpen || mMode != WRITE )
		MIRA_THROW(XIO, "Tape is not opened in write mode.");

	if (mMessageBlocks.size() > 0 ||
		mCurrentMessageBlock ||
		mMessages.size() > 0)
	{
		MIRA_THROW(XInvalidConfig,
		          "Waiting for altered start time is only supported "
		          "when no messages have been written to tape.");
	}

	mWaitForAlteredStartTime = true;
}

void Tape::alterStartTime(const Time& startTime)
{
	// check open mode
	if (!mIsOpen || mMode != WRITE )
		MIRA_THROW(XIO, "Tape is not opened in write mode.");
	if (mWaitForAlteredStartTime || 
		(mMessageBlocks.size() == 0 && 
		!mCurrentMessageBlock &&
		mMessages.size() == 0))
	{
		boost::mutex::scoped_lock lock(mMessageMutex);
		Duration timeOffset = mFile.start - startTime;
		setStartTime(startTime);
		// write all messages with corrected offset
		while (mMessages.size() > 0)
		{
			write(mMessages.begin()->first + timeOffset, mMessages.begin()->second);
			mMessages.erase(mMessages.begin());
		}
		mWaitForAlteredStartTime = false;
		// write the file header
		writeFileHeader();
	}
	else
		MIRA_THROW(XInvalidConfig, "Altering start time is only supported after "
		          "calling waitForAlteredStartTime() or when no messages "
		          "have been written to tape.");
}

void Tape::openInfo(const Path& file)
{
	Path resolved = resolvePath(file);
	if ( !boost::filesystem::exists(resolved) )
		MIRA_THROW(XFileNotFound, "Tape file '" << resolved.string()
		          << "' could not be found!");
	mFile.file.open(resolved.string().c_str(), std::ios::in | std::ios::binary);
	if(!mFile.file.is_open())
		MIRA_THROW(XIO,"Failed to open tape '" << file.string() << "' for reading")

	// read the file header
	readFileHeader();
	// read all channel informations
	readChannelInfo();
}

void Tape::openRead(const Path& file)
{
	openInfo(file);
	// read all message block informations
	readMessageBlocks();
}

void Tape::closeRead()
{
}

void Tape::openWrite(const Path& file)
{
	Path resolved = resolvePath(file);
	Path folder = resolved.parent_path();
	// create destination folder if not exists
	if (!boost::filesystem::exists(folder))
		boost::filesystem::create_directory(folder);
	mFile.file.open(resolved.string().c_str(), std::ios::out | std::ios::binary);
	if(!mFile.file.is_open())
		MIRA_THROW(XIO,"Failed to open tape '" << file.string() << "' for writing")

	mFile.filename = resolved.filename().string();
	mFile.version = getCurrentVersion();
	setStartTime(Time::now());
	mFile.nrBlocks = 0;
	mFile.nrChannels = 0;
	mFile.offsetToFirstInfo = 0;
	mLastInfo = 0;
	mWrittenChannels.clear();
	// write the file header
	writeFileHeader();
}

void Tape::closeWrite()
{
	// write all the data thats left in the sort window message map
	{
		boost::mutex::scoped_lock lock(mMessageMutex);

		foreach(const auto& message, mMessages)
			write(message.first, message.second);
	}
	// finish last message block if we haven't yet
	finishMessageBlock();
	// write the header again with updated data
	writeFileHeader();
	// write index tables

	// rewrite all channel info entries with updated data
	foreach(auto info, mChannels)
	{
		mFile.file.seekp(0, std::ios::end);
		info.second.offsetToIndexTable = mFile.file.tellp();
		Buffer<uint8> indexBuffer;
		BinaryBufferOstream ios(&indexBuffer);
		ios << (uint32)info.second.messages.size();
		for (uint32 i=0; i<info.second.messages.size(); ++i)
			ios << info.second.messages[i].block << 
				info.second.messages[i].offset << 
				info.second.messages[i].timeOffset.totalNanoseconds();

		writeHeader(INDEX, indexBuffer.size());
		mFile.file.write((char*)indexBuffer.data(), indexBuffer.size());
		mFile.file.seekp(info.second.offset+Tape::sHeaderSize+sizeof(uint64));
		Buffer<uint8> metaBuffer;
		BinaryBufferSerializer bs(&metaBuffer);
		bs.serialize(info.second.meta);
		bs.serialize(info.second.metaDB);
		Buffer<uint8> buffer;
		BinaryBufferOstream os(&buffer);
		os << info.second.firstMessageOffset.totalNanoseconds() <<
			info.second.lastMessageOffset.totalNanoseconds() <<
			info.second.name << info.second.type << metaBuffer << info.second.dataSize << info.second.offsetToIndexTable;
		mFile.file.write((char*)buffer.data(), buffer.size());
	}
}

void Tape::readFileHeader()
{
	mFile.file.seekg(0, std::ios::beg);

	mFile.file.read((char*)&mFile.version, sizeof(uint32));
	if (mFile.version > getCurrentVersion())
		MIRA_THROW(XIO, "Wrong tape file version should be <= "
		           << std::hex << getCurrentVersion() << " is " << mFile.version);
	uint64 time;
	mFile.file.read((char*)&time, sizeof(uint64));
	if (mFile.version < 0x00010001)
		time *= 1000000;
	mFile.start = Time::fromUnixNS(time);
	if (mFile.version > 0x00030000)
	{
		int64 tz;
		mFile.file.read((char*)&tz, sizeof(int64));
		mFile.timezoneOffset = Duration::nanoseconds(tz);
	}
	else
		mFile.timezoneOffset = mFile.start.toLocal() - mFile.start;
	mFile.file.read((char*)&mFile.nrBlocks, sizeof(uint32));
	if (mFile.version < 0x00020000)
	{
		uint32 dummy;
		mFile.file.read((char*)&dummy, sizeof(uint32));
		mFile.offsetToFirstInfo = dummy;
	}
	else
		mFile.file.read((char*)&mFile.offsetToFirstInfo, sizeof(uint64));
	if (mFile.file.fail())
		MIRA_THROW(XIO, "Could not read tape header. "
		           "This is not a tape file or tape is corrupt");
}

void Tape::writeFileHeader()
{
	mFile.file.seekp(0, std::ios::beg);
	Buffer<uint8> buffer;
	BinaryBufferOstream os(&buffer);
	os << mFile.version << mFile.start.toUnixNS() << mFile.timezoneOffset.totalNanoseconds()
		<< mFile.nrBlocks << mFile.offsetToFirstInfo;
	mFile.file.write((char*)buffer.data(), buffer.size());
}

Tape::Header Tape::readHeader()
{
	Header h;
	mFile.file.read((char*)&h.type, sizeof(char));
	mFile.file.read((char*)&h.size, sizeof(uint32));
	return h;
}

void Tape::writeHeader(HeaderType type, uint32 size)
{
	mFile.file.write((char*)&type, sizeof(char));
	mFile.file.write((char*)&size, sizeof(uint32));
}

void Tape::readChannelInfo()
{
	uint64 offset = mFile.offsetToFirstInfo;
	while (offset != 0)
	{
		mFile.file.seekg(offset);
		Header h = readHeader();
		if ( (HeaderType)h.type != CHANNELINFO )
			MIRA_THROW(XIO, "Expected channel info field has wrong format");
		Buffer<uint8> buffer(h.size);
		mFile.file.read((char*)buffer.data(), h.size);
		ChannelInfo info;
		readChannelInfoEx(buffer, mFile.version, offset, info);
		ChannelMap::iterator iiter = mChannels.insert(ChannelMap::value_type(info.name, info)).first;
		mFile.file.seekg(info.offsetToIndexTable);
		Header indexHeader = readHeader();
		if ( (HeaderType)indexHeader.type != INDEX )
			MIRA_THROW(XIO, "Expected index info field has wrong format");
		Buffer<uint8> indexBuffer(indexHeader.size);
		mFile.file.read((char*)indexBuffer.data(), indexHeader.size);
		BinaryBufferIstream iis(&indexBuffer);
		uint32 nrMessages;
		iis >> nrMessages;
		iiter->second.messages.resize(nrMessages);
		for (uint32 i=0; i<nrMessages; ++i)
		{
			int64 t;
			if (mFile.version < 0x00020000)
			{
				uint32 dBlock, dOffset;
				iis >> dBlock >> dOffset >> t;
				iiter->second.messages[i].block = dBlock;
				iiter->second.messages[i].offset = dOffset;
			}
			else
				iis >> iiter->second.messages[i].block >> iiter->second.messages[i].offset >> t;
			if (mFile.version < 0x00010001)
				t *= 1000;
			iiter->second.messages[i].timeOffset = Duration::nanoseconds(t);
		}
	}
}

void Tape::writeChannelInfo(ChannelInfo& info)
{
	mFile.file.seekp(0, std::ios::end);
	uint64 offset = mFile.file.tellp();
	// write offset to this entry into last entry (linking)
	// only if its not the first entry
	if (mFile.nrChannels++ > 0)
	{
		mFile.file.seekp(mLastInfo);
		mFile.file.write((char*)&offset, sizeof(uint64));
		mFile.file.seekp(0, std::ios::end);
	}
	else
		mFile.offsetToFirstInfo = offset;
	info.offset = offset;

	Buffer<uint8> metaBuffer;
	BinaryBufferSerializer bs(&metaBuffer);
	bs.serialize(info.meta);
	bs.serialize(info.metaDB);
	Buffer<uint8> buffer;
	BinaryBufferOstream os(&buffer);
	os << (uint64)0 << info.firstMessageOffset.totalNanoseconds() <<
		info.lastMessageOffset.totalNanoseconds() <<
		info.name << info.type << metaBuffer << info.dataSize << info.offsetToIndexTable;
	writeHeader(CHANNELINFO, buffer.size());
	mLastInfo = mFile.file.tellp();
	mFile.file.write((char*)buffer.data(), buffer.size());
	// increase size of message block by size of channel info
	mCurrentMessageBlock->size += Tape::sHeaderSize+buffer.size();
}

uint64 Tape::getFileHeaderSize() const
{
	uint64 headerSize = 0;
	headerSize += sizeof(uint32); // version
	headerSize += sizeof(uint64); // time of recording in nanoseconds since 00:00:00 01.01.1970
	if (mFile.version >= 0x00040000)
		headerSize += sizeof(int64); // time difference of recording machine to UTC time in nanoseconds (8)
	if (mFile.version >= 0x00020000)
		headerSize += sizeof(uint64); // offset to first channel info field
	else
		headerSize += sizeof(uint32); // offset to first channel info field
	headerSize += sizeof(uint32); // number of message blocks
	return headerSize;
}

void Tape::readMessageBlocks()
{
	uint64 offset = getFileHeaderSize();
	for (uint32 i=0; i<mFile.nrBlocks; ++i)
	{
		mFile.file.seekg(offset);
		Header h = readHeader();
		if ( (HeaderType)h.type != MESSAGEBLOCK )
			MIRA_THROW(XIO, "Expected message block nr. " << i << " has wrong format.");
		Buffer<uint8> buffer(h.size);
		mFile.file.read((char*)buffer.data(), h.size);
		BinaryBufferIstream is(&buffer);
		MessageBlock c;
		int64 firstMessageOffset;
		int64 lastMessageOffset;
		is >> c.nrMessages >> c.size >> firstMessageOffset >> lastMessageOffset;
		if (mFile.version < 0x00010001)
		{
			firstMessageOffset *= 1000;
			lastMessageOffset *= 1000;
		}
		c.firstMessageOffset = Duration::nanoseconds(firstMessageOffset);
		c.lastMessageOffset = Duration::nanoseconds(lastMessageOffset);
		c.offset = offset;
		mMessageBlocks.insert(MessageBlockMap::value_type(offset, c));
		offset = offset+c.size;
	}
}

void Tape::readMessageBlock(uint64 offset)
{
	MessageBlockMap::iterator mblock = mMessageBlocks.find(offset);
	if ( mblock == mMessageBlocks.end() )
		MIRA_THROW(XIO, "Tape does not contain message block at offset " << offset);

	// copy message block info to current message block
	mCurrentMessageBlock->offset = offset;
	mCurrentMessageBlock->nrMessages = mblock->second.nrMessages;
	mCurrentMessageBlock->size = mblock->second.size;
	mCurrentMessageBlock->firstMessageOffset = mblock->second.firstMessageOffset;
	mCurrentMessageBlock->lastMessageOffset = mblock->second.firstMessageOffset;
	// adapt buffer size of message block and read message block from file
	mCurrentMessageBlock->buffer.resize(mCurrentMessageBlock->size);
	mFile.file.seekg(offset+sHeaderSize+sMessageBlockSize);
	mFile.file.read((char*)mCurrentMessageBlock->buffer.data(), mCurrentMessageBlock->size);
}

void Tape::startMessageBlock(Duration time)
{
	mFile.file.seekp(0, std::ios::end);
	mCurrentMessageBlock = MessageBlock();
	mCurrentMessageBlock->offset = mFile.file.tellp();
	mCurrentMessageBlock->size = Tape::sHeaderSize+Tape::sMessageBlockSize;
	mCurrentMessageBlock->nrMessages = 0;
	mCurrentMessageBlock->firstMessageOffset = time;
	mCurrentMessageBlock->lastMessageOffset = time;
	Buffer<uint8> buffer;
	BinaryBufferOstream os(&buffer);
	os << mCurrentMessageBlock->nrMessages << mCurrentMessageBlock->size <<
		mCurrentMessageBlock->firstMessageOffset.totalNanoseconds() << 
		mCurrentMessageBlock->lastMessageOffset.totalNanoseconds();
	writeHeader(MESSAGEBLOCK, buffer.size());
	mFile.file.write((char*)buffer.data(), buffer.size());
}

void Tape::finishMessageBlock()
{
	if ( !mCurrentMessageBlock )
		return;
	// write the message block info again with updated data
	mFile.file.seekp(mCurrentMessageBlock->offset+Tape::sHeaderSize);
	Buffer<uint8> buffer;
	BinaryBufferOstream os(&buffer);
	os << mCurrentMessageBlock->nrMessages << mCurrentMessageBlock->size <<
		mCurrentMessageBlock->firstMessageOffset.totalNanoseconds() << 
		mCurrentMessageBlock->lastMessageOffset.totalNanoseconds();
	mFile.file.write((char*)buffer.data(), buffer.size());
	++mFile.nrBlocks;
	mMessageBlocks.insert(MessageBlockMap::value_type(mCurrentMessageBlock->offset,
	                                                  *mCurrentMessageBlock));
	mCurrentMessageBlock = boost::optional<MessageBlock>();
}


void Tape::readMessage(const MessageIndex& index, std::string& frameID,
                       uint32& sequenceID, Buffer<uint8>& data,
                       Duration& oTime)
{
	bool compressed;
	readMessage(index,frameID,sequenceID,data,oTime,compressed);
}

void Tape::readMessage(const MessageIndex& index, std::string& frameID,
                       uint32& sequenceID, Buffer<uint8>& data, Duration& oTime,
                       bool& oCompressed)
{
	// check open mode
	if ( !mIsOpen || mMode != READ )
		MIRA_THROW(XIO, "Tape is not opened in read mode");

	boost::mutex::scoped_lock lock(mMutex);
	// if we don't have a loaded message block or the index of the requested
	// message block does not match the current one read the whole message
	// block from file
	if ( !mCurrentMessageBlock )
	{
		mCurrentMessageBlock = MessageBlock();
		readMessageBlock(index.block);
	}
	if ( mCurrentMessageBlock->offset != index.block )
		readMessageBlock(index.block);

	// read header directly without using a bufferstream
	uint8* ptr = mCurrentMessageBlock->buffer.data() +
		(index.offset-sHeaderSize-sMessageBlockSize);
	if ( (HeaderType)(*ptr) != MESSAGE )
		MIRA_THROW(XIO, "Expected message at offset " << index.offset
		           << " in message block at offset " << index.block
		           << " has wrong format");
	uint32 messageHeaderSize = *reinterpret_cast<uint32*>(++ptr);

	// read message info located at message block offset + index offset - message block header
	Buffer<uint8> messageBuffer(mCurrentMessageBlock->buffer.data() +
	                            (index.offset-sMessageBlockSize), messageHeaderSize);
	BinaryBufferIstream messageStream(&messageBuffer);
	std::string name;

	uint32 compressedMessageSize;
	int64 timeOffset;
	messageStream >> timeOffset >> name;
	if (mFile.version >= 0x00050000)
		messageStream >> frameID >> sequenceID;
	messageStream >> compressedMessageSize >> oCompressed;
	if (mFile.version < 0x00010001)
		timeOffset *= 1000;
	oTime = Duration::nanoseconds(timeOffset);

	if (oCompressed)
	{
		uint32 messageSize;
		messageStream >> messageSize;
		Buffer<uint8> dataBuffer(mCurrentMessageBlock->buffer.data() +
		                         (index.offset-sMessageBlockSize+messageHeaderSize),
		                         compressedMessageSize);
		data.reserve(messageSize);
		uncompressBuffer(dataBuffer, data);
	}
	else
	{
		// read message TODO make faster?
		data.resize(compressedMessageSize);
		memcpy(data.data(), mCurrentMessageBlock->buffer.data() +
		       (index.offset-sMessageBlockSize+messageHeaderSize), data.size());
	}
	if (mFile.version < 0x00050000)
	{
		// extract stamped header from data and throw it away
		BinaryBufferIstream dataStream(&data);
		std::string tn;
		uint64 ts;
		dataStream >> tn >> ts >> frameID >> sequenceID;
		std::size_t headerSize = sizeof(uint32)+tn.size()		// size of serialized typename
								 + sizeof(uint64)				// size of serialized timestamp
								 + sizeof(uint32)+frameID.size()// size of serialized frame id
								 + sizeof(uint32);				// size of serialized sequence id
		data.pop_front(headerSize);
	}
}

///////////////////////////////////////////////////////////////////////////////

}
