/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteModule.C
 *    Implementation of RemoteModule.h.
 *
 * @author Tim Langner
 * @date   2010/11/16
 */

#include <fw/Framework.h>

#include <fw/RemoteModule.h>

#include <boost/array.hpp>
#include <boost/interprocess/sync/scoped_lock.hpp>

#include <communication/MulticastSender.h>
#include <stream/BinaryStream.h>
#include <thread/Atomic.h>

#include <fw/UnitManager.h>
#include <fw/FrameworkMessage.h>
#include <fw/FrameworkDefines.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

const Duration TryReconnectInterval = Duration::seconds(2);

///////////////////////////////////////////////////////////////////////////////

void RemoteModule::AuthSettings::clearAuth()
{
	mKeyFile.reset();
	mKey.reset();
	mPassword.reset();
}

const std::string& RemoteModule::AuthSettings::getPassword() const
{
	if(!mPassword)
		MIRA_THROW(XInvalidParameter, "No key authentication set");
	return *mPassword;
}

void RemoteModule::AuthSettings::setPassword(const boost::optional<std::string>& passwd)
{
	clearAuth();
	mPassword = passwd;
}

const RSAKey& RemoteModule::AuthSettings::getKey() const
{
	if(!mKey)
		MIRA_THROW(XInvalidParameter, "No key authentication set");
	return *mKey;
}

void RemoteModule::AuthSettings::setKey(const boost::optional<std::string>& str)
{
	clearAuth();
	std::string key = *str;
	std::string s = "PRIVATE:" + toString(key.size()/2) + ":" + key + ";";
	mKey = fromString<RSAKey>(s);
}

void RemoteModule::AuthSettings::setKeyFile(const boost::optional<std::string>& file)
{
	clearAuth();
	std::ifstream is(file->c_str());
	RSAKey key;
	is >> key;
	mKey = key;
}

///////////////////////////////////////////////////////////////////////////////


RemoteModule::RemoteModule() :
	mIncomingData(0),
	mOutgoingData(0),
	mCyclicRunnable(boost::bind(&RemoteModule::process, this),
	                Duration::milliseconds(100), "#RemoteModule"),
	mPort(0),
	mEnablePTPSync(true),
	mEnablePingTimeout(false),
	mPingInterval(Duration::seconds(1)),
	mPingTimeout(Duration::seconds(10)),
	mIOThreadCount(1),
	mExitOnDisconnectTimeout(Duration::invalid())
{
}

uint32 RemoteModule::getCurrentVersion()
{
	return MIRA_PROTOCOL_VERSION;
}

uint16 RemoteModule::getPort() const
{
	if (mServer)
		return mServer->getPort();
	return mPort;
}

void RemoteModule::setAuthGroup(const std::string& group)
{
	mAuthSettings.group = group;
}

void RemoteModule::setNoAuth()
{
	mAuthSettings.clearAuth();
}

void RemoteModule::setAuthPassword(const std::string& password)
{
	mAuthSettings.setPassword(password);
}

void RemoteModule::setAuthKey(const std::string& key)
{
	mAuthSettings.setKey(key);
}

void RemoteModule::setAuthKeyFile(const std::string& keyfile)
{
	mAuthSettings.setKeyFile(keyfile);
}

const RemoteModule::AuthSettings& RemoteModule::getAuthSettings() const
{
	return mAuthSettings;
}

ServiceLevel RemoteModule::getServiceLevel(const std::string& channelID,
                                           const Typename& channelType)
{
	if (mChannelServiceLevels.count(channelID) > 0)
		return mChannelServiceLevels[channelID];

	if (mChannelTypeServiceLevels.count(channelType) > 0)
		return ServiceLevel(channelID, mChannelTypeServiceLevels[channelType]);

	return ServiceLevel(channelID);
}

void RemoteModule::start()
{
	mID = boost::uuids::random_generator()();

	// resolve all names of channels that use service level agreements
	// (can only be done here, since at the time mChannelServiceLevels
	// is deserialized the config file is not fully loaded)
	foreach(const auto& serviceLevel, mServiceLevels)
	{
		std::string channelID = MIRA_FW.getNameRegistry().resolve(serviceLevel.channelID, "/");
		mChannelServiceLevels[channelID] = serviceLevel;
		mChannelServiceLevels[channelID].channelID = channelID;
	}
	foreach(const auto& serviceLevel, mTypeServiceLevels)
		mChannelTypeServiceLevels[serviceLevel.channelType] = serviceLevel;

	mServer.reset(new RemoteServer(mPort));
	mServer->run(mIOThreadCount, false);

	MIRA_LOG(DEBUG) << "Framework: Started remote framework server ID='"
		<< mID << "' on port " << getPort() << " , running " << mIOThreadCount << " threads";

	mDiscoverService.reset();
	if(MIRA_CMDLINE.getOptions().count("autodiscover"))
	{
		try	
		{
			mDiscoverService.reset(new DiscoverService());
		} catch (...)
		{
			MIRA_LOG(WARNING) << "Failed to start Framework Discovery Service";
			mDiscoverService.reset();
		}
		MIRA_LOG(DEBUG) << "Auto discovery of remote frameworks is enabled";
	}
	MIRA_LOG(DEBUG) << "Starting remote frameworks cyclic runnable: " << &mCyclicRunnable;
	mThread = boost::thread(boost::bind(&CyclicRunnable::operator(), &mCyclicRunnable));

	if(mDiscoverService)
	{
		DiscoverService::AnnounceMessage data;
		data.port = getPort();
		data.id = mID;
		MulticastSender announceService(MIRA_AUTODISCOVERY_MULTICAST_ADDRESS,
		                                MIRA_AUTODISCOVERY_MULTICAST_PORT);
		announceService.send(std::string((const char*)&data, (const char*)&data+sizeof(DiscoverService::AnnounceMessage)));
	}
}

void RemoteModule::stop()
{
	if (mDiscoverService)
		mDiscoverService->stop();

	mThread.interrupt();
	mThread.join();

	if(mServer)
		mServer->stop();

	while (!mConnections.empty())
	{
		{
			boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
			mConnections.begin()->second->stop(); // also moves the connection to 'stopped connections' in the pool
		}
		MIRA_SLEEP(50)
	}
	std::vector<ConnectionList*> pendingConnections{&mPendingOutgoingConnections, &mPendingIncomingConnections};
	foreach (ConnectionList* list, pendingConnections)
	{
		while (!list->empty())
		{
			{
				boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
				(*list->begin())->stop(); // also moves the connection to 'stopped connections' in the pool
			}
			MIRA_SLEEP(50)
		}		
	}

	mRemoteConnectionPool.releaseStoppedConnections();
}

ScopedAccess<const RemoteModule::ConnectionMap, boost::recursive_mutex>
RemoteModule::getConnections() const
{
	return ScopedAccess<const RemoteModule::ConnectionMap,
	                    boost::recursive_mutex>(&mConnections, &mConnectionMutex);
}

void RemoteModule::addKnownFramework(const std::string& address)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	KnownFramework f;
	f.address = address;
	mKnownFrameworks.push_back(f);
}

void RemoteModule::addKnownFramework(const std::string& address,
                                     bool forcePTP, bool legacyBinaryFormat,
                                     bool monitorOnly,
                                     const Duration& delayConnect)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	KnownFramework f;
	f.address = address;
	f.forcePTP = forcePTP;
	if (legacyBinaryFormat)
		f.binaryFormatVersion = 0;
	f.monitorOnly = monitorOnly;
	f.lastConnectionTry = Time::now() - TryReconnectInterval + delayConnect;
	mKnownFrameworks.push_back(f);
}

bool RemoteModule::isFrameworkConnected(const std::string& frameworkID) const
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(const auto& connection, mConnections)
		if (connection.second->getFrameworkID() == frameworkID)
			return true;
	return false;
}

bool RemoteModule::isConnectedTo(const std::string& address) const
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(const auto& connection, mConnections)
		if (connection.second->getAddress().address == address)
			return true;
	return false;
}

void RemoteModule::disconnectFramework(const std::string& frameworkID,
                                       bool autoReconnect)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(auto& connection, mConnections)
		if (connection.second->getFrameworkID() == frameworkID)
		{
			connection.second->setAutoReconnect(autoReconnect);
			connection.second->stop();
			break; // make sure to break the foreach loop, since iterators became invalid
		}
}

void RemoteModule::disconnectFrom(const std::string& address,
                                  bool autoReconnect)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(auto& connection, mConnections)
		if (connection.second->getAddress().address == address)
		{
			connection.second->setAutoReconnect(autoReconnect);
			connection.second->stop();
			break; // make sure to break the foreach loop, since iterators became invalid
		}
}

void RemoteModule::migrateUnitToThisFramework(const std::string& id)
{
	// Unit runs already in this framework
	if (MIRA_FW.getUnitManager()->getUnit(id))
		return;

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->hasAuthority(id))
		{
			c.second->migrateUnit(id);
			break;
		}
	}
}

void RemoteModule::publishAuthority(const AuthorityDescription& authority)
{
	RemoteConnection::AuthorityDescriptions authorities;
	authorities.push_back(authority);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->isSynchronized() )
			c.second->publishAuthorities(authorities);
	}
}

void RemoteModule::unpublishAuthority(const AuthorityDescription& authority)
{
	RemoteConnection::AuthorityDescriptions authorities;
	authorities.push_back(authority);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->isSynchronized() )
			c.second->unpublishAuthorities(authorities);
	}
}

void RemoteModule::publishService(const std::string& service)
{
	// someone in our local framework has published a service
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: Publishing service to remote frameworks service='"
		<< service << "'";

	std::set<std::string> services;
	services.insert(service);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->isSynchronized() )
			c.second->publishServices(services);
	}
}

void RemoteModule::unpublishService(const std::string& service)
{
	// someone in our local framework has unpublished a service
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: UnPublishing service to remote frameworks service='"
		<< service << "'";

	std::set<std::string> services;
	services.insert(service);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->isSynchronized() )
			c.second->unpublishServices(services);
	}
}

void RemoteModule::publishChannel(const std::string& channelID, const Typename& type)
{
	// someone in our local framework has published a channel
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: Publishing channel to remote frameworks channel='"
		<< channelID << "'";

	typedef std::map<std::string,Typename> ChannelTypeMap;
	ChannelTypeMap channels;
	channels[channelID] = type;

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->isSynchronized() )
			c.second->publishChannels(channels);
	}
}

void RemoteModule::unpublishChannel(const std::string& channelID)
{
	// no one in our local framework has published this channel any longer
	// notify all connected frameworks about this
	MIRA_LOG(DEBUG) << "Framework: No more publishers for channel='"
		<< channelID << "' exist.";

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (c.second->isSynchronized() )
			c.second->unpublishChannel(channelID);
	}
}

void RemoteModule::subscribeChannel(const std::string& channelID)
{
	// someone in our local framework has subscribed to a channel
	// search in all connected remote frameworks for a publisher of this
	// channel and subscribe there
	MIRA_LOG(DEBUG) << "Framework: Looking for remote publishers of channel '"
		<< channelID << "'";
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ServiceLevel serviceLevel(channelID);
	try {
		Typename channelType = MIRA_FW.getChannelManager().getTypename(channelID);
		serviceLevel = getServiceLevel(channelID, channelType);
		if (!serviceLevel.codecs.empty()) {
			JSONSerializer ts(false);
			JSONValue val = ts.serialize(serviceLevel.codecs);
			MIRA_LOG(NOTICE) << "Framework: Using codecs (service level) for subscribing "
				<< "to channel '" << channelID << "': " << json::write(val, false);
		}
	}
	catch(Exception& ex) {}
	foreach(ConnectionMap::value_type& c, mConnections)
		if (MIRA_FW.getChannelManager().hasPublished(c.second->getGlobalID(),
		                                             channelID))
			c.second->subscribeChannel(channelID, serviceLevel);
}

void RemoteModule::unsubscribeChannel(const std::string& channelID)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	foreach(ConnectionMap::value_type& c, mConnections)
	{
		if (MIRA_FW.getChannelManager().hasPublished(c.second->getGlobalID(),
		                                             channelID))
		{
			FrameworkMessageHeader header(channelID.length(), UNSUBSCRIBE_CHANNEL_MSG);
			boost::array<boost::asio::const_buffer, 2> buffers =
			{{
				header.asBuffer(),
				boost::asio::buffer(channelID)
			}};
			c.second->write(buffers);
		}
	}
}

void RemoteModule::updateIncomingStats(std::size_t size)
{
	mIncomingData += size;
}

void RemoteModule::updateOutgoingStats(std::size_t size)
{
	mOutgoingData += size;
}

std::size_t RemoteModule::getIncomingBytesPerSecond() const
{
	return mIncomingStats.dataPerSecond;
}

std::size_t RemoteModule::getOutgoingBytesPerSecond() const
{
	return mOutgoingStats.dataPerSecond;
}

RemoteConnectionProxy RemoteModule::createIncomingConnection()
{
	return mRemoteConnectionPool.createRemoteIncomingConnection();
}

RemoteConnectionProxy& RemoteModule::addPendingIncomingConnection(RemoteConnectionProxy&& proxy,
                                                                  bool start)
{
	mPendingIncomingConnections.push_back(std::move(proxy));
	RemoteConnectionProxy& inserted = mPendingIncomingConnections.back();
	if (start) inserted->start();
	return inserted;
}

void RemoteModule::removePendingIncomingConnection(RemoteConnection* connection)
{
	mPendingIncomingConnections.remove_if(
		[connection](const RemoteConnectionProxy& proxy) { return proxy == connection; });
}

RemoteConnectionProxy& RemoteModule::addPendingOutgoingConnection(RemoteConnectionProxy&& proxy,
                                                                  bool start)
{
	mPendingOutgoingConnections.push_back(std::move(proxy));
	RemoteConnectionProxy& inserted = mPendingOutgoingConnections.back();
	if (start) inserted->start();
	return inserted;
}

void RemoteModule::removePendingOutgoingConnection(RemoteConnection* connection)
{
	mPendingOutgoingConnections.remove_if(
		[connection](const RemoteConnectionProxy& proxy) { return proxy == connection; });
}

void RemoteModule::onRemoteFrameworkDiscovered(const std::string& host, uint16 port, UUID id)
{
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ConnectionMap::iterator i = mConnections.find(id);
	// do we have already a connection to that framework
	if (i == mConnections.end())
	{
		MIRA_LOG(DEBUG) << "Framework: Discovered remote framework ID='"
			<< id << "' at " << host << ":" << port;
		// no we have not -> store it!
		KnownFramework f;
		f.address = MakeString() << host << ":" << port;
		f.keep = false;
		mKnownFrameworks.push_back(f);
	}
}

void RemoteModule::storeConnection(RemoteConnectionProxy connection)
{
	boost::shared_ptr<PropertyNode> properties = MIRA_FW.getProperties();
	RootPropertyNode* root = properties->getRootNode();
	boost::interprocess::scoped_lock<RootPropertyNode> lock(*root);
	mConnections[connection->getRemoteID()] = std::move(connection);
	properties->synchronize();
}

void RemoteModule::eraseConnection(ConnectionMap::iterator it)
{
	boost::shared_ptr<PropertyNode> properties = MIRA_FW.getProperties();
	RootPropertyNode* root = properties->getRootNode();
	boost::interprocess::scoped_lock<RootPropertyNode> lock(*root);
	mConnections.erase(it);
	properties->synchronize();
}

bool RemoteModule::onIncomingConnected(RemoteIncomingConnection* connection)
{
	stopDisconnectTimeout();
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ConnectionMap::iterator i = mConnections.find(connection->getRemoteID());
	// do we have already a connection to that framework?
	if (i == mConnections.end())
	{
		MIRA_LOG(NOTICE) << "Framework: We have an incoming connection "
			"from remote framework at " << connection->getAddress().address
			<< " ID='" << connection->getFrameworkID() << "'";
		// no we have not -> store it! -> move from pending to connected
		auto it = std::find(mPendingIncomingConnections.begin(), mPendingIncomingConnections.end(), connection);

		if(it == mPendingIncomingConnections.end())
			MIRA_THROW(XConnection, "RemoteModule::onIncomingConnected: connection not found in pending incoming connections");
		storeConnection(std::move(*it));
		mPendingIncomingConnections.erase(it);
		connection->setAuthority(std::make_unique<Authority>("/", "IncomingConnection",
		                                                     Authority::ANONYMOUS |
		                                                     Authority::INTERNAL |
		                                                     Authority::INVISIBLE_PUBLISHER_SUBSCRIBER));

		connection->onConnect(isPTPSyncEnabled(), isPingTimeoutEnabled());
		return true;
	}
	else
	{
		MIRA_LOG(DEBUG) << "Framework: We have an incoming connection "
			"from an already connected remote framework ID='"
			<< connection->getFrameworkID() << "'";
		// yes we have -> disconnect
		return false;
	}
}

void RemoteModule::onIncomingDisconnected(RemoteIncomingConnection* connection)
{
	const KnownFramework& kfw = connection->getAddress();
	{
		boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
		if (!connection->getRemoteID().is_nil())
		{
			ConnectionMap::iterator i = mConnections.find(connection->getRemoteID());
			if (i == mConnections.end())
			{
				// we do not have a connection to this id
				// so put back the address to the known framework list
				removePendingIncomingConnection(connection);
				if (kfw.keep && !kfw.address.empty())
					mKnownFrameworks.push_back(kfw);
			}
			else
			{
				if (i->second == connection)
				{
					// we are the connection -> erase us and add address to the known framework list
					eraseConnection(i);
					if (kfw.keep && !kfw.address.empty())
						mKnownFrameworks.push_back(kfw);
					MIRA_LOG(NOTICE) << "Framework: A client from remote framework ID='"
						<< connection->getFrameworkID() << "' has disconnected from our server";
				}
			}
		}
		else
		{
			removePendingIncomingConnection(connection);
			// put back the address to the known framework list
			if (kfw.keep && !kfw.address.empty())
				mKnownFrameworks.push_back(kfw);
		}
	}

	startDisconnectTimeout();
}

bool RemoteModule::onOutgoingConnected(RemoteOutgoingConnectionBase* connection)
{
	stopDisconnectTimeout();
	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
	ConnectionMap::iterator i = mConnections.find(connection->getRemoteID());
	// do we have already a connection to that framework
	if (i == mConnections.end())
	{
		MIRA_LOG(DEBUG) << "Framework: We have connected to remote framework at "
			<< connection->getAddress().address << " ID='" << connection->getFrameworkID() << "'";
		// no we have not -> move from pending to connected!
		auto it = std::find(mPendingOutgoingConnections.begin(), mPendingOutgoingConnections.end(), connection);

		if(it == mPendingOutgoingConnections.end())
			MIRA_THROW(XConnection, "RemoteModule::onOutgoingConnected: connection not found in pending outgoing connections");
		storeConnection(std::move(*it));
		mPendingOutgoingConnections.erase(it);
		connection->setAuthority(std::make_unique<Authority>("/", "OutgoingConnection",
		                                                     Authority::ANONYMOUS |
		                                                     Authority::INTERNAL |
		                                                     Authority::INVISIBLE_PUBLISHER_SUBSCRIBER));

		connection->onConnect(isPTPSyncEnabled(), isPingTimeoutEnabled());
		return true;
	}
	else
	{
		MIRA_LOG(DEBUG) << "Framework: We have connected to an already "
			"connected remote framework at " << connection->getAddress().address << " ID='"
			<< connection->getFrameworkID() << "'";
		// yes we have -> copy the address to put it back to known framework list after disconnect
		i->second->setAddress(connection->getAddress());
		return false;
	}
}

void RemoteModule::onOutgoingDisconnected(RemoteOutgoingConnectionBase* connection)
{
	const KnownFramework& kfw = connection->getAddress();
	{
		boost::recursive_mutex::scoped_lock lock(mConnectionMutex);
		// remove from pending
		if (!connection->getRemoteID().is_nil())
		{
			ConnectionMap::iterator i = mConnections.find(connection->getRemoteID());
			if (i == mConnections.end())
			{
				removePendingOutgoingConnection(connection);
				// we do not have a connection to this id
				// so put back the address to the known framework list
				if (kfw.keep)
					mKnownFrameworks.push_back(kfw);
			}
			else
			{
				if (i->second == connection)
				{
					// we are the connection -> erase us and add address to the known framework list
					eraseConnection(i);
					if (kfw.keep)
						mKnownFrameworks.push_back(kfw);
					MIRA_LOG(DEBUG) << "Framework: We have disconnected from a remote "
						"framework at " << kfw.address << " ID='"
						<< connection->getFrameworkID() << "'";
				}
			}
		}
		else
		{
			removePendingOutgoingConnection(connection);
			// put back the address to the known framework list
			if (kfw.keep)
				mKnownFrameworks.push_back(kfw);
		}
	}

	startDisconnectTimeout();
}

void RemoteModule::process()
{
	// destroy stopped connections in a well-defined thread context
	mRemoteConnectionPool.releaseStoppedConnections();

	if (checkDisconnectTimeout())
		return;

	// update statistics
	mIncomingStats.update(Time::now(), mIncomingData);
	atomic::write(&mIncomingData, 0);
	mOutgoingStats.update(Time::now(), mOutgoingData);
	atomic::write(&mOutgoingData, 0);

	boost::recursive_mutex::scoped_lock lock(mConnectionMutex);

	if (mKnownFrameworks.empty())
		return;

	// The framework we will try to connect to next
	KnownFramework address = mKnownFrameworks.front();
	// Remove it from list
	mKnownFrameworks.pop_front();
	// If last connection try was not at least some seconds ago add address back and
	// continue
	if ((Time::now() - address.lastConnectionTry) < TryReconnectInterval)
	{
		mKnownFrameworks.push_back(address);
		return;
	}
	// Update last connection time
	address.lastConnectionTry = Time::now();
	try
	{
		addPendingOutgoingConnection(mRemoteConnectionPool.createRemoteOutgoingConnection(address));
	}
	catch(XInvalidParameter& ex)
	{
		MIRA_LOG(ERROR) << ex.message() << " Will not try to connect again!";
		// schedule termination if flag is set
		startDisconnectTimeout();
		return;
	}
	catch(Exception& ex)
	{
		MIRA_LOG(WARNING) << ex.message() << " Retrying...";
		// schedule termination if flag is set
		startDisconnectTimeout();
		mKnownFrameworks.push_back(address);
		return;
	}
}

void RemoteModule::startDisconnectTimeout()
{
	if (mExitOnDisconnectTimeout.isValid() && !mDisconnectedSince.isValid())
		mDisconnectedSince = Time::now();
}

void RemoteModule::stopDisconnectTimeout()
{
	mDisconnectedSince = Time::invalid();
}

bool RemoteModule::checkDisconnectTimeout()
{
	if (!mDisconnectedSince.isValid())
		return false;

	if (Time::now() - mDisconnectedSince < mExitOnDisconnectTimeout)
		return false;

	if (MIRA_FW.isTerminationRequested())
		return true;

	MIRA_LOG(ERROR) << "Exiting framework after connection was closed / connection attempts failed for too long!";
	MIRA_FW.requestTermination();

	return true;
}

///////////////////////////////////////////////////////////////////////////////

}
