/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkTransformer.C
 *    Implementation of FrameworkTransformer.h.
 *
 * @author Erik Einhorn
 * @date   2010/10/11
 */

#include <fw/Framework.h>

#include <fw/FrameworkTransformer.h>

#include <error/Logging.h>


namespace mira {

///////////////////////////////////////////////////////////////////////////////

FrameworkTransformer::FrameworkTransformer(Authority& authority) :
	mAuthority(authority)
{
	mAuthority.publishService(*this);
	mAuthority.registerCallbackForInterface("IFrameworkTransformer",
	                                        &FrameworkTransformer::registerInterface,
	                                        this);
}

FrameworkTransformer::~FrameworkTransformer()
{}

///////////////////////////////////////////////////////////////////////////////

void FrameworkTransformer::addLink(const std::string& childID,
                                   const std::string& parentID)
{
	addLink(childID, parentID, FrameworkTransformerNode::UNSPECIFIED);
}

void FrameworkTransformer::addLink(const std::string& childID,
                                   const std::string& parentID,
                                   FrameworkTransformerNode::Type type)
{
	if(createLink(childID, parentID, type)) {
		// if link was not already known, then synchronize all known transformers
		auto l = mAuthority.queryServicesForInterface("IFrameworkTransformer");
		foreach(const std::string& otherTransformer, l) {
			if(otherTransformer!=mAuthority.getID())
				notifyTransformerOfLink(otherTransformer,
				                        childID, parentID, type);
		}
	}
}

void FrameworkTransformer::storeTransformTree(const Path& filename)
{
	XMLDom xml;
	foreach(IDToNodeMap::value_type& n, mNodes)
	{
		if (n.second->getParent())
		{
			XMLDom::sibling_iterator i = xml.root().add_child("link");
			i.add_attribute("parent", n.second->getParent()->getID());
			i.add_attribute("child", n.second->getID());
			switch (static_cast<NodePtr>(n.second)->getType()) {
				case FrameworkTransformerNode::UNSPECIFIED: break;
				case FrameworkTransformerNode::MOVING     : i.add_attribute("type", "moving"); break;
				case FrameworkTransformerNode::FIXED      : i.add_attribute("type", "fixed"); break;
				default: break;
			}
		}
	}

	xml.saveToFile(filename);
}

void FrameworkTransformer::registerInterface(const std::string& interface,
                                             const std::string& otherTransformer)
{
	assert(interface=="IFrameworkTransformer");
	MIRA_LOG(DEBUG) << "Found other IFrameworkTransformer: " << otherTransformer;

	bool implementsLinkType = mAuthority.implementsInterface(otherTransformer,
	                                                         "IFrameworkTransformerLinkType");

	// tell the other transformer all of our links
	auto links = getLinks();
	foreach(auto& link, links) // links is a list of (parent,child) pairs
	{
		AbstractNodePtr parent = link.first;
		FrameworkTransformerNode* child = static_cast<NodePtr>(link.second);
		notifyTransformerOfLink(otherTransformer, implementsLinkType,
		                        child->getID(), parent->getID(),
		                        child->getType());
	}
}

bool FrameworkTransformer::createLink(const std::string& childID, const std::string& parentID,
                                      FrameworkTransformerNode::Type type)
{
	MIRA_LOG(DEBUG) << "Adding transform link: " << childID << " <- " << parentID
	                << " (Type: " << toString(type) << ")";

	// create/retrieve child and parent
	NodePtr child  = createNode(childID);
	NodePtr parent = createNode(parentID);

	bool linkAdded = Base::addLink(child, parent); // true -> parent initially set or changed

	if (type == FrameworkTransformerNode::UNSPECIFIED)
		return linkAdded;

	if (child->getType() == FrameworkTransformerNode::UNSPECIFIED)
		setNodeType(child, type);
	else {
		if (type != child->getType()) {
			if (linkAdded)
				setNodeType(child, type);
			else
				MIRA_LOG(WARNING) << "Link " << childID << " <- " << parentID << " already exists "
				                  << "with type " << toString(child->getType())
				                  << ", new type " << toString(type) << " is ignored.";
		}
	}

	return linkAdded;
}

void FrameworkTransformer::notifyTransformerOfLink(const std::string& otherTransformer,
                                                   const std::string& childID, const std::string& parentID,
                                                   FrameworkTransformerNode::Type type)
{
	notifyTransformerOfLink(otherTransformer,
	                        mAuthority.implementsInterface(otherTransformer, "IFrameworkTransformerLinkType"),
	                        childID, parentID, type);

}

void FrameworkTransformer::notifyTransformerOfLink(const std::string& otherTransformer,
                                                   bool implementsLinkType,
                                                   const std::string& childID, const std::string& parentID,
                                                   FrameworkTransformerNode::Type type)
{
	if (type > FrameworkTransformerNode::UNSPECIFIED) {
		if (implementsLinkType) {
			try {
				mAuthority.callService<void>(otherTransformer, "addLink",
				                             childID, parentID, type);
			}
			catch(XRPC& ex) {
				MIRA_LOG(WARNING) << "FrameworkTransformer '" << otherTransformer << "' implements IFrameworkTransformerLinkType "
				                  << "but addLink(... , type) failed, trying addTypedLink().";
				// try 'addTypedLink' fallback, this method was added in r9548 and renamed to 'addLink' in r9833
				mAuthority.callService<void>(otherTransformer, "addTypedLink",
				                             childID, parentID, type);
			}
			return;
		} else {
			MIRA_LOG(WARNING) << "FrameworkTransformer '" << otherTransformer << "' does not implement link type, "
			                  << "will ignore type " << toString(type) << " on node " << childID << ".";
		}
	}

	mAuthority.callService<void>(otherTransformer, "addLink",
	                             childID, parentID);
}

FrameworkTransformer::NodePtr FrameworkTransformer::createNode(const std::string& ID)
{
	boost::mutex::scoped_lock lock(mMutex);

	NodePtr node = getNode(ID);

	if(!node) {
		node = new FrameworkTransformerNode(ID, mAuthority);
		this->addNode(node);
		node->mChannel = mAuthority.subscribe<void>(ID);
	}

	return node;
}

void FrameworkTransformer::setNodeType(NodePtr node, FrameworkTransformerNode::Type type)
{
	node->setType(type);
	if (type == FrameworkTransformerNode::FIXED)
		MIRA_FW.getChannelManager().setMaxSlots(node->getID(), 1);
}

///////////////////////////////////////////////////////////////////////////////

RigidTransform2f FrameworkTransformer::getTransform2(const std::string& target,
                                                     const std::string& source)
{
	return mAuthority.getTransform<RigidTransform2f>(target, source, Time::now());
}

RigidTransformCov2f FrameworkTransformer::getTransformCov2(const std::string& target,
                                                           const std::string& source)
{
	return mAuthority.getTransform<RigidTransformCov2f>(target,source, Time::now());
}

RigidTransform3f FrameworkTransformer::getTransform3(const std::string& target,
                                                     const std::string& source)
{
	return mAuthority.getTransform<RigidTransform3f>(target, source, Time::now());
}

RigidTransformCov3f FrameworkTransformer::getTransformCov3(const std::string& target,
                                                           const std::string& source)
{
	return mAuthority.getTransform<RigidTransformCov3f>(target, source, Time::now());
}

void FrameworkTransformer::publishTransform2(const std::string& nodeID,
                                             const RigidTransform2f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

void FrameworkTransformer::publishTransformCov2(const std::string& nodeID,
                                                const RigidTransformCov2f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

void FrameworkTransformer::publishTransform3(const std::string& nodeID,
                                             const RigidTransform3f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

void FrameworkTransformer::publishTransformCov3(const std::string& nodeID,
                                                const RigidTransformCov3f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

///////////////////////////////////////////////////////////////////////////////

// the specialized versions of FrameworkTransformReadAccessor and
// FrameworkTransformWriteAccessor are not defined inline in the header to
// reduce the compilation time dramatically

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, NearestNeighborInterpolator>::
get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, NearestNeighborInterpolator&) {
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolator>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolator& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolatorNoExtrapolation>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolatorNoExtrapolation& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolatorNearestNeighbourExtrapolator>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolatorNearestNeighbourExtrapolator& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolatorExtrapolationLimit>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolatorExtrapolationLimit& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

// explicit instantiation
template struct FrameworkTransformReadAccessor<RigidTransform2f,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,NearestNeighborInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolatorNoExtrapolation>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolatorNearestNeighbourExtrapolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolatorExtrapolationLimit>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,NearestNeighborInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolatorNoExtrapolation>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolatorNearestNeighbourExtrapolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolatorExtrapolationLimit>;


template <typename Transform>
void FrameworkTransformWriteAccessor<Transform>::set(Authority& authority, const std::string& nodeID,
                                                     const Transform& transform, const Time& time)
{
	Channel<Transform> channel = authority.publish<Transform>(nodeID);
	ChannelWrite<Transform> data = channel.write();

	std::string frameID;

	// determine node's parent as frame id (if it is a node and has a parent)
	FrameworkTransformer::NodePtr n = MIRA_FW.getTransformer()->getNode(nodeID);
	if (n) {
		AbstractTransformerNode* p = n->getParent();
		if (p)
			frameID = p->getID();
	}

	data = makeStamped(transform, time, frameID);
}

template struct FrameworkTransformWriteAccessor<RigidTransform2f>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov2f>;
template struct FrameworkTransformWriteAccessor<RigidTransform3f>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov3f>;
template struct FrameworkTransformWriteAccessor<RigidTransform2d>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov2d>;
template struct FrameworkTransformWriteAccessor<RigidTransform3d>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov3d>;

///////////////////////////////////////////////////////////////////////////////

}
