/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapePlayer.h
 *    Class for playing back tapes.
 *
 * @author Tim Langner
 * @date   2010/12/29
 */

#ifndef _MIRA_TAPEPLAYER_H_
#define _MIRA_TAPEPLAYER_H_

#include <serialization/Serialization.h>
#include <fw/Framework.h>
#include <fw/TapeVisitor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * Class for playing back tape files
 */
class TapePlayer
{
public:

	TapePlayer();

	virtual ~TapePlayer()
	{
		stop();
		mAuthority.checkout();
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{}

	/**
	 * Pauses the play back
	 */
	void pause()
	{
		mStepTo.reset();
		mStepToChannel.reset();
		mJumpTo.reset();
		mPlayImmediately = false;
		mPause = true;
	}

	/**
	 * Is the playback paused
	 * @return true if paused
	 */
	bool isPaused() const
	{
		return mPause;
	}

	/**
	 * return if we are currently in stepTo mode
	 */
	bool inStepTo() const
	{
		return mStepTo.is_initialized() || mStepToChannel.is_initialized();
	}

	/**
	 * Do a single step. Will set pause mode.
	 * Jumps to the next chronological message to visit
	 */
	void step()
	{
		stepTo(Duration::microseconds(-1));
	}

	/**
	 * Do a step by playing back all messages between current message time and time.
	 * Stopping again at time, setting pause mode.
 	 * If time is above last message time in tape, this will just
 	 * start/resume normal playback (same as play()).
	 */
	void stepTo(Duration time);

	/**
	 * Do a step by playing back all messages between current message time and
	 * the time of the next message in channel channelID.
	 * Stopping again at next message time in channel channelID, setting pause mode.
	 * By default, returns true;
 	 * If no such channel or no more messages for it exist, this will just
 	 * go to the next message immediately and start/resume normal playback (same as play()),
 	 * then return false.
	 */
	bool stepTo(const std::string& channelID);

	/**
	 * Do a step by playing back all messages between current message time and
	 * the time of the next message in any of the channels in channelIDs.
	 * Stopping again at next message time in any of the specified channels,
	 * setting pause mode.
	 * By default, return true.
 	 * If no such channel or no more messages for any of them exist, this will just
 	 * just go to the next message immediately and start/resume normal playback
 	 * (same as play()), then return false;
	 */
	bool stepToAnyOf(const std::vector<std::string>& channelIDs);

	/**
	 * Stops play back
	 */
	void stop()
	{
		mThread.interrupt();
		mThread.join();
		mIsPlaying = false;
		mErrorMessage.reset();
	}

	/**
	 * Turn the loop mode on/off
	 * @param[in] loop Should the tape be played in loop mode
	 * @param[in] count How often should the tape be played back in loop mode
	 *            (values < 0 = infinite loops (default))
	 */
	void setLoop(bool loop, int32 count = -1)
	{
		mLoopCount = count;
		mLoop = loop;
	}

	/**
	 * Set the time scaling factor for play back.
	 * @param[in] scaler The scale factor 
	 *  - 1 means normal play back, 
	 *  - 2 means double speed, 
	 *  - 0.5 means half speed
	 */
	void setTimeScaler(float scaler)
	{
		assert(scaler >= 0.0f);
		mTimeScaler = 1.0f/scaler;
	}

	/**
	 * Get the time scaling factor
	 * @return Time scaling factor used for playback
	 */
	float getTimeScaler() const
	{
		return 1.0f/mTimeScaler;
	}

	/**
	 * Prepares playing back data from the visitor.
	 * @param[in] iVisitor The visitor used for visiting the messages.
	 * @param[in] startTime The tape is played back relatively to this time.
	 * @param[in] prefix A namespace prefix. This prefix will be appeded to all channel names.
	 */
	void load(TapeVisitor* iVisitor, Time startTime=Time::now(), const std::string& prefix = "");

	/**
	 * Starts playing back the messages.
	 */
	void play();

	/**
	 * Are we currently in playback mode (i.e. a file is opened for playback)
	 * @return true if playing
	 */
	bool isPlaying() const
	{
		return mIsPlaying;
	}

	/**
	 * When an error occurs during play back this method will return true.
	 * In case of an error the play back is stopped automatically and the
	 * isPlaying() method will return false. In this case the
	 * error message can be obtained via getErrorMessage()
	 */
	bool hasError() const
	{
		return mErrorMessage.is_initialized();
	}

	/**
	 * Returns the error message of an error that occurred during play back.
	 */
	std::string getErrorMessage() const
	{
		if (mErrorMessage)
			return *mErrorMessage;
		return "";
	}

	/**
	 * Get the recorded time offset of the currently played or to be played message
	 * (if currently paused the current message will be the one that is played after
	 * resume)
	 * @return Recorded time offset of the current message in microseconds
	 */
	Duration getCurrentMessageTimeOffset() const
	{
		if (mVisitor == NULL)
			return Duration::nanoseconds(0);
		if (mMessage == mVisitor->end())
			return mVisitor->getLastMessageTimeOffset();
		return mMessage.getTimeOffset();
	}

	/**
	 * Get the recorded time zone offset of the tape containing the currently played or
	 * to be played message (if currently paused the current message will be the one
	 * that is played after resume)
	 * @return Recorded time zone offset of the current message in microseconds
	 */
	Duration getCurrentMessageTZOffset() const
	{
		if (mVisitor == NULL)
			return Duration::nanoseconds(0);
		if (mMessage == mVisitor->end())
			return mVisitor->getLastMessageTZOffset();
		return mMessage.getTZOffset();
	}

	/**
	 * Returns the relative time the player plays back messages to.
	 * The real message time is defined as relative start time + message time offset. 
	 */
	Time getRelativeStartTime() const
	{
		return mStartTime;
	}

	/**
	 * Jump to a given message at recorded time offset.
	 * Jumps to the message with an offset closest to time.
	 * @param[in] time The time offset of the message in microseconds we like
	 *            to jump to.
	 */
	void jumpTo(Duration time);

	/**
	 * Return the authority of the player.
	 */
	Authority* getAuthority()
	{
		return &mAuthority;
	}

protected:

	void stepTo(const TapeVisitor::iterator& i);
	void doPlay();

protected:

	Authority mAuthority;
	boost::thread mThread;
	int32 mLoopCount;
	bool mLoop;
	bool mPause;
	bool mPlayImmediately;
	bool mIsPlaying;
	float mTimeScaler;
	Time mStartTime;
	boost::optional<Duration> mJumpTo;
	boost::optional<Duration> mStepTo;
	boost::optional<TapeVisitor::iterator> mStepToChannel;
	boost::optional<std::string> mErrorMessage;
	TapeVisitor* mVisitor;
	TapeVisitor::iterator mMessage;
	boost::mutex mConditionMutex;
	boost::condition_variable mCondition;
	std::string mNamespacePrefix;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
