/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ResourceName.h
 *    Class for resolving and managing names and namespaces.
 *
 * @author Erik Einhorn
 * @date   2010/09/16
 */

#ifndef _MIRA_RESOURCENAME_H_
#define _MIRA_RESOURCENAME_H_

#include <string>
#include <ostream>
#include <istream>

#include <error/Exceptions.h>

#include <serialization/IsTransparentSerializable.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Class for storing/combining/managing resource names consisting of namespaces
 * and names separated by '/'
 * Resource names can be combined and normalized.
 */
class ResourceName
{
private:

	/// Only for internal use
	ResourceName(const std::string& name, bool dontNormalize) : mName(name) {}

public:
	/// Creates default empty resource name
	ResourceName() {}

	/// Creates a normalized resource from a name 
	ResourceName(const std::string& name) : mName(normalize(name)) {
		if(!validate(mName))
			MIRA_THROW(XInvalidParameter, 
			           "Trying to create ResourceName from invalid name: '"
			           << mName << "'");
	}

	/// Variant for normal C-Strings
	ResourceName(const char* name) : mName(normalize(std::string(name))) {
		if(!validate(mName))
			MIRA_THROW(XInvalidParameter, 
			           "Trying to create ResourceName from invalid name: '"
			           << mName << "'");
	}

	ResourceName(const ResourceName& other) : mName(other.mName) {}

public:

	/// Assignment for strings
	const ResourceName& operator=(const std::string& other) {
		mName = normalize(other);
		return *this;
	}

	const ResourceName& operator=(const ResourceName& other) {
		mName = other.mName;
		return *this;
	}

	/// Casts ResourceName into std::string
#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40601
	operator std::string() const { return mName; }
# else
	operator const std::string&() const { return mName; }
# endif
#else
	// this should be the preferred version, that is supported by GCC < 4.6.1 and VC 2010
	operator const std::string&() const { return mName; } 
#endif

	/// Returns the underlying string containing the name
	const std::string& str() const { return mName; }

public:

	/**
	 * Returns parent namespace.
	 * If *this has no parent (e.g. "namewithoutparent"), it returns an empty
	 * ResourceName.
	 */
	ResourceName parent() const;

	/// Returns the leaf, i.e. the bottom most name (the string after the last / ).
	std::string leaf() const;

	/// Returns true, if this node is an ancestor of "other" in the namespace hierarchy.
	bool isAncestorOf(const ResourceName& other) const;

	/// Returns true, if this node is the direct ancestor (parent) of "other" in the namespace hierarchy.
	bool isParentOf(const ResourceName& other) const;

	/// Returns true, if this node is a successor of "other" in the namespace hierarchy.
	bool isSuccessorOf(const ResourceName& other) const;

	/// Returns true, if this node is the direct successor (child) of "other" in the namespace hierarchy.
	bool isChildOf(const ResourceName& other) const;

public:

	/** @name Comparison operators */
	//@{
	bool operator<(const ResourceName& other) const  { return mName <  other.mName; }
	bool operator<=(const ResourceName& other) const { return mName <= other.mName; }
	bool operator>(const ResourceName& other) const  { return mName >  other.mName; }
	bool operator>=(const ResourceName& other) const { return mName >= other.mName; }
	bool operator==(const ResourceName& other) const { return mName == other.mName; }
	bool operator!=(const ResourceName& other) const { return mName != other.mName; }
	//@}

public:

	/// Returns the i-th character in the name
	std::string::value_type operator[](std::string::size_type i) { return mName[i]; }

public:
	template<typename Reflector>
	void reflect(Reflector& r) {
		r.delegate(mName);
	}

	/**
	 * Concatenates two resource names separated by a '/'
	 */
	std::string operator/(const std::string& other) const {
		if(isRoot()) // we are the root namespace
			return mName + other;

		// concatenate both names
		return mName + "/" + other;
	}

	/**
	 * Concatenates two resource names separated by a '/'
	 */
	ResourceName operator/(const ResourceName& other) const {
		return ResourceName(*this / other.mName, true);
	}

	/// Concatenates this with another resource names separated by a '/'
	const ResourceName& operator/=(const ResourceName& other) {
		if(isRoot())
			mName += other.mName;
		else
			mName += "/" + other.mName;
		return *this;
	}

	/**
	 * Returns true, if this is the root namespace "/"
	 */
	bool isRoot() const {
		return mName.size()==1 && mName[0]=='/';
	}

	/**
	 * Returns true, if the name is fully qualified.
	 */
	bool isFullyQualified() const {
		if(mName.empty())
			return false;

		return mName[0]=='/';
	}

public:

	static ResourceName makeFullyQualified(const ResourceName& name,
	                                       const ResourceName& ns);

public:

	friend std::ostream& operator<<(std::ostream& os, const ResourceName& name)
	{
		os << name.mName;
		return os;
	}

	friend std::istream& operator>>(std::istream& is, ResourceName& name)
	{
		std::string s;
		is >> s;
		name = s;
		return is;
	}

private:

	static std::string normalize(const std::string& name);

	/// checks if specified name is valid and returns true, if name is valid
	static bool validate(const std::string& name);

private:
	/// our name (Invariant: will always be normalized!!)
	std::string mName;
};

///////////////////////////////////////////////////////////////////////////////

template<typename SerializerTag>
class IsTransparentSerializable<ResourceName,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif
