/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DispatcherThread.h
 *    Dispatcher class that allows registration of handler functions 
 *    that need to be called from within the same thread.
 *
 * @author Erik Einhorn
 * @date   2010/11/21
 */

#ifndef _MIRA_DISPATCHERTHREAD_H_
#define _MIRA_DISPATCHERTHREAD_H_

#include <set>
#include <deque>

#ifndef Q_MOC_RUN
#include <boost/function.hpp>
#include <boost/thread/condition_variable.hpp>
#endif

#include <error/LoggingAux.h>
#include <utils/Time.h>
#include <utils/StringAlgorithms.h>
#include <thread/Thread.h>
#include <fw/Status.h>
#include <fw/Runnable.h>

namespace mira {


template<typename T, typename Sequence = std::vector<T>,
         typename Compare  = std::less<typename Sequence::value_type> >
class IteratablePriorityQueue : public std::priority_queue<T,Sequence,Compare>
{
	typedef std::priority_queue<T,Sequence,Compare> Base;
public:

	// types
	typedef typename Sequence::iterator iterator;
	typedef typename Sequence::const_iterator const_iterator;

public:
	explicit IteratablePriorityQueue(const Compare& x, const Sequence& s) :
		Base(x,s) {}

	explicit IteratablePriorityQueue(const Compare& x = Compare(), Sequence&& s = Sequence()) :
		Base(x,std::move(s)) {}


	template<typename InputIterator>
	IteratablePriorityQueue(InputIterator first, InputIterator last, const Compare& x, const Sequence& s) :
		Base(first,last,x,s) {}

	template<typename InputIterator>
	IteratablePriorityQueue(InputIterator first, InputIterator last, const Compare& x, Sequence&& s) :
		Base(first,last,x,std::move(s)) {}

public:

	iterator begin() { return this->c.begin(); }
	iterator end() { return this->c.end(); }

	const_iterator begin() const { return this->c.begin(); }
	const_iterator end() const { return this->c.end(); }


};





///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * Class that can be used whenever you want to have ONE thread where several
 * handlers are assigned to. The dispatcher waits for a certain signal
 * and processes all handlers afterwards.
 * Moreover you can assign timers that are called periodically
 * or tasks that are called upon their invocation time, all from within the
 * same thread.
 *
 * This class can be used together with boost::thread by binding the run()
 * method to a boost thread:
 * /code
 * DispatcherThread dispatcher;
 * ...
 * boost::thread t(boost::bind(&Dispatcher::run(), &dispatcher)) can be used.
 * /endcode
 */
class DispatcherThread
{
public:

	/// forward declaration
	class Timer;
	/// Signature of a timer/task callback function
	typedef boost::function<void(const Timer&)> TimerCallback;

	/**
	 * Class representing timers and tasks that can be registered and executed
	 * by the dispatcher thread. Timers have an invocation time when they will
	 * be executed again. Timers will be rescheduled after invocation based on
	 * their period. Tasks form a kind of single shot timer that will be
	 * executed once at a given time.
	 */
	class Timer : public IRunnable, boost::noncopyable
	{
	protected:
		/// Only dispatcher thread can create us
		friend class DispatcherThread;

		/**
		 * Constructs a timer with a given callback that is called whenever
		 * the timers invocation time is due and a period in which the timer
		 * is rescheduled after invocation.
		 */
		Timer(TimerCallback callback, Duration period,
		      Duration tolerance = Duration::invalid());

		/**
		 * Constructs a one shot timer or task with a given callback that is
		 * called once at the timers/tasks invocation time and a specified
		 * time.
		 */
		Timer(TimerCallback callback, Time invokationTime);

	protected:

		/**
		 * Called by dispatcher when timer is rescheduled
		 */
		void updateInvocationTime() {
			currentExpected += mPeriod;
		}

		void run(DispatcherThread* dispatcher) override;

	public:

		/**
		 * Returns if the timer is active.
		 */
		bool isActive() const { return mActive; }

		/**
		 * Changes the period of the timer. New period will be used next time
		 * the timer gets rescheduled.
		 */
		void setPeriod(Duration period) { mPeriod = period; }

		/**
		 * Returns the current period of the timer.
		 */
		Duration getPeriod() const { return mPeriod; }

		/**
		 * Set the tolerance that is allowed to exceed the next invocation time
		 * before issuing a warning and resetting the invocation time.
		 * If invalid then there is no limit for exceeding the time.
		 */
		void setTolerance(Duration tolerance) { mTolerance = tolerance; }

		/**
		 * Get the tolerance for exceed the next invocation time.
		 * Invalid if not specified.
		 */
		Duration getTolerance() const { return mTolerance; }

		/**
		 * Checks and returns the amount of time the timer has exceeded
		 * the expected invocation time.
		 * Returns 0 for timers that have 0 as period or if the exceeded time is
		 * inside the specified tolerance.
		 */
		Duration getExceedance() const
		{
			if (mPeriod > Duration::seconds(0))
			{
				Duration dt = current - currentExpected;
				if (dt > mTolerance)
					return dt;
			}
			return Duration::seconds(0);
		}


	public:

		/**
		 * Start the timer (activates it)
		 */
		void start();

		/**
		 * Stops a timer (deactivates it)
		 */

		void stop();
		/**
		 * Return next time of execution
		 */
		Time getNextInvocationTime() const {
			return currentExpected;
		}

	public:

		/// time the last callback happened
		Time last;

		/// time the current callback should be happening
		Time currentExpected;

		/// time the current callback was actually called (Time::now() as of the beginning of the callback)
		Time current;

		/// How long the last callback ran for
		Duration lastDuration;

	private:

		TimerCallback mCallback;
		Duration mPeriod;
		Duration mTolerance;
		bool mOneShot;
		bool mActive;
	};

	typedef boost::shared_ptr<Timer> TimerPtr;

public:

	/**
	 * Construct a dispatcher thread with an optional name.
	 * Name will be used when registering the dispatcher at the thread monitor.
	 */
	DispatcherThread(const std::string& name = "");

	virtual ~DispatcherThread();

	void setName(const std::string& name);

	bool insertRunnable(IRunnablePtr runnable, bool singleton = false, Time time=Time::now());
	void removeRunnable(IRunnablePtr runnable);


	/**
	 * Adds a runnable that is executed once as soon as possible within the
	 * DispatcherThreads main thread.
	 * If the immediate handler throws an exception, the DispatcherThread enters
	 * an unrecoverable error state. Additionally, a diagnostics module can
	 * be specified where the error is reported.
	 */
	void addImmediateHandler(IRunnablePtr runnable);

	/**
	 * Adds a function that is executed once as soon as possible within the
	 * DispatcherThreads main thread.
	 * If the immediate handler throws an exception, the DispatcherThread enters
	 * an unrecoverable error state. Additionally, a diagnostics module can
	 * be specified where the error is reported.
	 */
	template <typename F>
	void addImmediateHandlerFunction(F&& fn, DiagnosticsModulePtr errorModule=nullptr) {
		IRunnablePtr r(new FunctionRunnable<F>(static_cast<F&&>(fn),errorModule));
		addImmediateHandler(r);
	}

	/**
	 * Adds a runnable that is executed once just before the thread terminates.
	 */
	void addFinalizeHandler(IRunnablePtr runnable);

	/**
	 * Adds a function that is executed once just before the thread terminates.
	 * Additionally, a diagnostics module can be specified where the error
	 * is reported if there is any.
	 */
	template <typename F>
	void addFinalizeHandlerFunction(F&& fn, DiagnosticsModulePtr errorModule=nullptr) {
		IRunnablePtr r(new FunctionRunnable<F>(static_cast<F&&>(fn),errorModule));
		addFinalizeHandler(r);
	}

	/**
	 * Creates and adds a timer that gets called cyclic in a given period
	 * and has a tolerance for exceeding the given period.
	 * (if oneshot is false) and calls the given callback on every invocation.
	 * @return pointer to created timer. Can be used for removeTimer().
	 */
	TimerPtr createTimer(Duration period, Duration periodTolerance,
	                     TimerCallback callback, bool oneshot=false);

	/**
	 * Creates and adds a timer that gets called cyclic in a given period
	 * (if oneshot is false) and calls the given callback on every invocation.
	 * @return pointer to created timer. Can be used for removeTimer().
	 */
	TimerPtr createTimer(Duration period, TimerCallback callback,
	                     bool oneshot=false);

	/**
	 * Creates a timer that calls callback exactly once at the given time.
	 * @return pointer to created timer. Can be used for removeTimer().
	 */
	TimerPtr createTimer(Time time, TimerCallback callback);

	/**
	 * Removes the given timer (timer callback will not be called again and
	 * timer will not be rescheduled). Calling start on a removed timer will
	 * not start it again.
	 */
	void removeTimer(TimerPtr timer);

	/**
	 * Returns true, if the given timer
	 */
	bool hasTimer(TimerPtr timer);

	/**
	 * Starts the dispatcher.
	 * If the dispatcher is running already this method does nothing.
	 * The dispatcher will call all init handlers first and removes them
	 * afterwards. This ensures, that when stopping and starting the dispatcher
	 * again, the init handlers are not called again.
	 * If one of the init handlers fails, the dispatcher enters an unrecoverable
	 * error state and cannot be started again, since the state of the entity
	 * that failed to initialize is undefined.
	 *
	 * Usually the dispatcher starts its own process thread, however, if you
	 * set the optional parameter to false, no thread will be started. Instead,
	 * you need to call the spin() method on your own.
	 *
	 * @throw XLogical If a previous initialization has failed, and the
	 *                 dispatcher cannot be started again.
	 */
	void start(bool startThread = true);

	/**
	 * Stops the dispatcher, if it is running. This method blocks, until the
	 * dispatcher thread (if there is any) has terminated and was joined.
	 */
	void stop();

	/**
	 * Returns true, if there was a failure while processing the immediate
	 * handlers.
	 * This is an unrecoverable failure that will disallow the further
	 * execution of the dispatcher.
	 */
	bool hasUnrecoverableFailure() const;

	/**
	 * Exception that can be thrown in every handler to indicate an
	 * unrecoverable failure.
	 */
	class XUnrecoverableFailure {};

	/// Returns true, if the dispatcher currently is running
	bool isRunning() const;

	/**
	 * Invoke the dispatcher manually (instead of starting a separate thread
	 * via the start() method). spin() must be called periodically, to allow
	 * the dispatcher to work properly.
	 * The caller should stop calling spin(), when false is returned.
	 *
	 * Usually the spin() method waits for events and signals of the dispatcher,
	 * you can specify the max. time that is spent in the call using the
	 * maxWait parameter. If this parameter is set to 0, spin() will not block
	 * at all. If it is set to another duration, it will wait until the next
	 * signal or event or stop request or the specified wait duration is
	 * reached.
	 * @param[in] maxWait maximum wait time for new events
	 */
	bool spin(Duration maxWait = Duration::infinity());

	/**
	 * Returns the thread id of the thread that is used to process the
	 * dispatcher. This usually will be the dispatcher thread that is started
	 * in the start() method. However, if no thread was launched in the start()
	 * method, this method returns the id of the thread that is calling the
	 * spin() method.
	 */
	boost::thread::id getThreadID() const {
		return mThreadId;
	}

	/**
	 * Checks if there is a work item (either an immediate handler or an item in the execution queue with an
	 * execution time before horizon).
	 * One application of this member function is to call spin() in a test setting till every job is done.
	 *
	 * @param horizon specifies the interval [-inf, horizon] to check for work. If horizon is invalid, this
	 * method only checks the immediate handler.
	 *
	 * @return True if there are immediate handlers. If horizon is valid, true if there are any queue items
	 *         with time <= horizon. False otherwise.
	 */
	[[nodiscard]] bool hasWork(Time horizon = Time::now()) const;

protected:
	/**
	 * Runs this thread dispatchers main loop.
	 * This will invoke all init handlers immediately after the thread is
	 * launched. Afterwards the thread will wait for a signal. If the signal
	 * arrives the signal handlers are called. Additionally, the timers
	 * after their invocation time is reached.
	 */
	void run();

private:

	void runThread();

private:

	/// returns true if processing the init handlers was successful, false in an error case
	bool processImmediateHandlers();
	bool processFinalizeHandlers();
	bool processSpin(const Duration& maxWait);

private:

	bool isInterruptionRequested();

protected:

	/// XXX
	struct QueueItem
	{
		QueueItem() {}
		QueueItem(IRunnablePtr r, Time t) : runnable(r), time(t), toBeRemoved(false) {}


		IRunnablePtr runnable;
		Time time;
		bool toBeRemoved;

		bool operator<(const QueueItem& other) const { return time>other.time; }
	};

	typedef IteratablePriorityQueue<QueueItem, std::deque<QueueItem>> Queue;


	Queue mQueue;
	std::set<IRunnablePtr> mPendingRunnables;

	mutable boost::mutex mConditionMutex;
	boost::condition_variable mCondition;

	///////////////////////////////////////////////////////////////////////////////

	void postProcessTimer(TimerPtr timer);

	///////////////////////////////////////////////////////////////////
	std::string mName;

	std::list<IRunnablePtr> mImmediateHandlers;
	mutable boost::mutex mImmediateHandlerMutex;

	std::list<IRunnablePtr> mFinalizeHandlers;
	boost::mutex mFinalizeHandlerMutex;

	std::set<TimerPtr> mTimers;
	boost::mutex mTimerMutex; // protects above handler sets

	bool mUnrecoverableFailure;
	bool mIsRunning;

	// unfortunatelly, we need our own interruption requested flag, since
	// boost::this_thread::interruption_requested() is cleared after a
	// boost::thread_interrupted exception is thrown, however, we need this
	// flag to be true, until we really have terminated
	bool mInterruptionRequested;

	boost::thread mThread;
	boost::thread::id mThreadId;
	bool mPendingSignal;

};

typedef DispatcherThread::Timer Timer;
typedef DispatcherThread::TimerPtr TimerPtr;
typedef DispatcherThread::TimerCallback TimerCallback;

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_DISPATCHERTHREAD_H_ */
