/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelSubscriber.h
 *    Subscriber classes used to subscribe on channels for automatic
 *    notification.
 *
 * @author Erik Einhorn
 * @date   2010/09/08
 */

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#endif

#include <utils/Bind.h>

#include <fw/AbstractChannel.h>
#include <fw/AbstractChannelSubscriber.h>

#include <fw/Status.h>


#ifndef _MIRA_CHANNELSUBSCRIBER_H_
#define _MIRA_CHANNELSUBSCRIBER_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward declaration
template<typename T>
class ConcreteChannel;

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * Implements  @ref AbstractChannelSubscriber for a concrete data type.
 *
 * For internals on the signaling process see @ref Channel.
 *
 * @see Channel
 */
template<typename T, typename ChannelRead>
class ChannelSubscriber : public AbstractChannelSubscriber
{
public:

	typedef ConcreteChannel<T> ChannelType;

public:

	ChannelSubscriber(DispatcherThread* dispatcher,
	                  boost::function<void (ChannelRead)> invokeFn,
	                  DiagnosticsModulePtr diagnosticsModule = NULL) :
		AbstractChannelSubscriber(dispatcher,diagnosticsModule),
		mChannel(NULL),
		mInvokeFn(invokeFn),
		mLastTimestamp(Time::invalid()) {}

	/// Returns the channel this subscriber is attached to
	AbstractChannelPtr getChannel() { return mChannel; }

	/**
	 * Attaches this subscriber to the specified channel.
	 * This is called from Channel::addSubscriber.
	 */
	void attachChannel(ChannelType* channel) {
		assert(mChannel==NULL);
		mChannel=channel;
	}

	/**
	 * Detaches this subscriber from the channel.
	 */
	virtual void detachChannel() {
		mChannel=NULL;
	}


	// implements IRunnable
	virtual void run(DispatcherThread* dispatcher)
	{
		if(!mEnabled)
			return;

		// prevents this Runnable instance from being destroyed while we are in here
		IRunnablePtr This = this->shared_from_this();

		assert(mChannel!=NULL);
		std::string channelid = mChannel->getID();

		try {

			// read the latest data from the Channel and call the invoke
			// function while securing that it was not already invoked with
			// the same data (timestamp)
			auto r = mChannel->read();
			Time n = r.getTimestamp();
			if (mLastTimestamp != n)
			{
				mLastTimestamp = n;
				mInvokeFn(r);
			}

			if(mDiagnosticsModule)
				mDiagnosticsModule->ok(channelid);
		}
		catch(Exception& ex) {
			if (!mDiagnosticsModule || mDiagnosticsModule->error(channelid,
			           ex.message(), "Exception while invoking subscriber")) {
				MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
			}
		}
		catch(std::exception& ex) {
			if (!mDiagnosticsModule || mDiagnosticsModule->error(channelid,
			            ex.what(), "Exception while invoking subscriber")) {
				MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
			}
		}
	}


private:

	ChannelType* mChannel;
	boost::function<void (ChannelRead)> mInvokeFn;
	Time mLastTimestamp;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
