/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelVectorSynchronizer.h
 *    Filter for synchronized callbacks on subscribed channels.
 *
 * @author Christoph Weinrich, Ronny Stricker
 * @date   2014/04/14
 */

#ifndef _MIRA_CHANNELVECTORSYNCHRONIZER_H_
#define _MIRA_CHANNELVECTORSYNCHRONIZER_H_

#ifndef Q_MOC_RUN
#include <boost/preprocessor/seq.hpp>
#include <boost/preprocessor/tuple/elem.hpp>
#include <boost/preprocessor/repetition/repeat.hpp>
#include <boost/preprocessor/comparison/not_equal.hpp>
#include <boost/preprocessor/repetition/for.hpp>
#include <boost/preprocessor/comma_if.hpp>
#include <boost/preprocessor/arithmetic/inc.hpp>

#include <boost/tuple/tuple.hpp>
#endif

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#ifdef DOXYGEN
// This is a fake class to generate a doxygen documentation
/**
 * Class that can be registered as a filter when subscribing to more than 
 * one channel to only get a callback when all channels have data with 
 * a similar timestamp (differ only in the specified tolerance).
 * All channels have to be of the same type.
 * To synchronize channels of differnt types, you have to use the
 * ChannelVectorSynchronizer2, ChannelVectorSynchronizer3, ...
 * template class. To synchronize arbitrary different types of
 * channels, just define MIRA_CHANNEL_SYNCHRONIZER(MySynchronizerName, 5).
 * Where MySynchronizerName defines the name of the template class which will
 * be generated, the number will define the number of channels, which the
 * synchronizer is using. After that you can use your MySynchronizerName class
 * exactly the same way.
 *
 * \code
 * MyUnit : public MicroUnit
 * {
 * 		void initialize()
 * 		{
 * 			std::vector<std::string> mChannelNames;
 * 			mChannelNames.push_back("IntChannel0");
 * 			mChannelNames.push_back("IntChannel1");
 * 			mChannelNames.push_back("IntChannel2");
 * 			myFilter.subscribe(*this, mChannelNames,
 * 			                   &MyUnit::onCallback, this, Duration::milliseconds(500));
 * 		}
 *
 * 		void onCallback(std::vector<ChannelRead<int>> channelReads)
 * 		{}
 *
 * 		ChannelVectorSynchronizer<int> myFilter;
 * }
 * \endcode
 */

template<typename type0> class ChannelVectorSynchronizer1	{
public:

	/* generate constructor which is empty*/
	ChannelVectorSynchronizer1()
	{}

	/**
	 * Call this instead of Authority::unsubscribe(...) to unsubscribe
	 * all the channels of the synchronizer
	 */
	void unsubscribe(Authority& authority)
	{
		while(mChannel0.size() != 0){
			authority.unsubscribe<type0>(mChannel0.back().getID());
			mLast0.pop_back();
			mChannelRead0.pop_back();
			mChannel0.pop_back();
		}
	}

	/**
	 * Call this instead of Authority::subscribe()
	 * Function provided for convenience (if no callback function is required).
	 */
	void subscribe(Authority& authority,
			 const std::vector<std::string>& channelID0
			,const Duration& t = Duration::seconds(0))
	{
		mFunction = 0;
		for(const auto& channelID:(channelID0))
		{
			(mLast0).push_back(Time::unixEpoch());
			(mChannelRead0).push_back(ChannelRead<type0>());
			(mChannel0).push_back(authority.subscribe<type0>(channelID, &ChannelVectorSynchronizer1<
				 type0>::callback0, this, t));
		}
		mTolerance = t;
	}

	/**
	 * Call this instead of Authority::subscribe()
	 * @param authority The authority that does the basic subscription
	 * @param channelIDA The ID of the channel A we are subscribing on
	 * @param channelIDB The ID of the channel B we are subscribing on
	 * @param fn The function that should be called when the filter matches
	 * @param t Maximum time difference of the channels message timestamps for
	 *          still being synchronous
	 */
	void subscribe(Authority& authority,
			 const std::vector<std::string>& channelID0
			,boost::function<void ( std::vector<ChannelRead<type0>>)> fn
			,const Duration& t = Duration::seconds(0))
	{
		mFunction = fn;
		for(const auto& channelID:(channelID0))
    {
		(mLast0).push_back(Time::unixEpoch());
		(mChannelRead0).push_back(ChannelRead<type0>());
		(mChannel0).push_back(authority.subscribe<type0>(channelID, &ChannelVectorSynchronizer1<
				 type0>::callback0, this, t));
	}		mTolerance = t;
	}

	/**
	 * Same as above but with a function and object pointer. Provided for convenience.
	 */
	template<typename Class>
	void subscribe(Authority& authority,
			 const std::vector<std::string>& channelID0
			, void (Class::*f)(  std::vector<ChannelRead<type0>> )
			, Class* obj
			, const Duration& t = Duration::seconds(0)
			)
	{
		subscribe(authority,
				 channelID0,
				boost::bind(f, obj,  _1),t);
	}

	// Called when vector of channels of type0 has new data
	void callback0(ChannelRead<type0> ic) {
		std::vector<ChannelRead<type0>> c0;

		try{
			for(auto& channel:mChannel0){
				c0.push_back(channel.read(ic.getTimestamp(), mTolerance));
			};
		}
		catch(...)
		{ return; }
		/* now we can call the function with all channel elements */
		call( c0);
	}

	void call( std::vector<ChannelRead<type0>> c0)
	{
		typename std::vector<ChannelRead<type0>>::const_iterator cIter0 = c0.begin();
		std::vector<Time>::iterator lastIter0 = mLast0.begin();
		for(; cIter0 != c0.end(); ++cIter0, ++lastIter0){
			if(*lastIter0 == cIter0->getTimestamp())
				return;
		}
		cIter0 = c0.begin();
		lastIter0 = mLast0.begin();
		for(; cIter0 != c0.end(); ++cIter0, ++lastIter0){
			*lastIter0 = cIter0->getTimestamp();
		}
		mChannelRead0 = c0;

		if (mFunction)
			mFunction(c0);
	}

	/**
	 * Return (synchronized) ChannelRead objects.
	 * The ChannelRead objects are always synchronized by the channel with the lowest
	 * update frequency. Please note that the ChannelRead objects may be invalid if
	 * no data has been published yet.
	 */
	boost::tuple< std::vector<ChannelRead<type0>>> read() {
		/* BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_ISVALID,mChannelRead)*/
		if(!isValid()){
			MIRA_THROW(XInvalidRead, "Cannot read from synchronized channel "\
						<< "' since there is no data");\
		};
		return boost::make_tuple( mChannelRead0);
	}

	/**
	 * Return true if all ChannelRead objects contain valid data.
	 */
	bool isValid() const {
		for(const auto& channelRead:mChannelRead0) {
			if(!channelRead.isValid()){
				return false;
			}
		}
		return true;
	}

	/**
	 * Return the latest (synchronized) element once it is available.
	 * If the functions times out, invalid channelRead objects will be returned.
	 * Please note that the function does not regard synchronized elements that
	 * were available BEFORE this function was called.
	 */
	boost::tuple< std::vector<ChannelRead<type0>>> waitForData(const Duration& timeout = Duration::infinity())
	{
		/* validateReadAccess is private :( */
		/*BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_VALIDATE,mChannel)*/

		Time end;
		if(!timeout.isValid() || timeout.isInfinity())
			end = Time::eternity();
		else
			end = Time::now() + timeout;

		while(!boost::this_thread::interruption_requested())
		{
			Time oldestTimeStamp = Time::now();
			Time timeStamp;
			try {
				std::vector<ChannelRead<type0>> c0;
				for(auto& channel:mChannel0){
					timeStamp = channel.read().getTimestamp();
					if ( timeStamp < oldestTimeStamp )
						oldestTimeStamp = timeStamp;
				}
				for(auto& channel:mChannel0){
					c0.push_back(channel.read(oldestTimeStamp, mTolerance));
				};
				mChannelRead0 = c0;
				return read();
			}
			catch(XInvalidRead& ex) {}
			if(Time::now()>end) /* handle timeout */
				break;
			mira::sleep(mira::Duration::milliseconds(50));;
		}
		return boost::make_tuple( mChannelRead0);
	}

	bool waitForPublisher(const Duration& timeout = Duration::infinity()) const
	{
		/*return BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER,);*/
		/* bool result = true; */
		for(const auto& channel:mChannel0){
			/* if(result) */
			/*	result = channel.waitForPublisher( timeout ); */
			if(!channel.waitForPublisher( timeout ))
				return false;
		}
		/* return result; */
		return true;
	}

	std::vector<Time> mLast0;
	std::vector<Channel<type0>> mChannel0;
	std::vector<ChannelRead<type0>> mChannelRead0;
	Duration mTolerance;
	boost::function<void ( std::vector<ChannelRead<type0>>)> mFunction;
};


#else

///////////////////////////////////////////////////////////////////////////////

// create: data0, data1, ...
#define MIRA_FW_INTERNAL_NUMBER(z, n, data) \
		BOOST_PP_COMMA_IF(n) BOOST_PP_CAT(data,n)

// create: data1, data2, ...
#define MIRA_FW_INTERNAL_INC_NUMBER(z, n, data) \
		BOOST_PP_COMMA_IF(n) BOOST_PP_CAT(data,BOOST_PP_INC(n))

// create: std::vector<ChannelRead<type0>>, std::vector<ChannelRead<type1>>, ...
#define MIRA_FW_INTERNAL_CHANNEL(z, n, data) \
		BOOST_PP_COMMA_IF(n) std::vector<ChannelRead<BOOST_PP_CAT(type,n)>>

// create: std::vector<ChannelRead<type0>> c0, std::vector<ChannelRead<type1>> c1, ...
#define MIRA_FW_INTERNAL_CHANNEL_VAR(z, n, data) \
		BOOST_PP_COMMA_IF(n) std::vector<ChannelRead<BOOST_PP_CAT(type,n)>> BOOST_PP_CAT(c,n)

// create:
// foreach(const auto& channel, mChannel(n)){
//   if(!channel.waitForPublisher( timeout ))
//     return false;
// }
#define MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER(z, n, data) \
		foreach(const auto& channel, BOOST_PP_CAT(mChannel,n)){ \
			if(!channel.waitForPublisher( timeout )) \
				return false; \
		} \

// create:
// foreach(const auto& channelRead, data(n)){
//   if(!channelRead.isValid())
//     return false;
// }
#define MIRA_FW_INTERNAL_ISVALID(z,n,data) \
		foreach(const auto& channelRead, BOOST_PP_CAT(data,n)) { \
			if(!channelRead.isValid()){ \
				return false; \
			} \
		} \

// create:
// std::vector<Time> mLast(n);
// std::vector<Channel<type(n)>> mChannel(n);
// std::vector<ChannelRead<type(n)>> mChannelRead(n);
#define MIRA_FW_INTERNAL_MEMBER(z, n, data) \
	std::vector<Time> BOOST_PP_CAT(mLast,n);\
	std::vector<Channel<BOOST_PP_CAT(type,n)>> BOOST_PP_CAT(mChannel,n);\
	std::vector<ChannelRead<BOOST_PP_CAT(type,n)>> BOOST_PP_CAT(mChannelRead,n);\

// create:
// std::vector<ChannelRead<type(n)>> c(n);
#define MIRA_FW_INTERNAL_DECL_CHANNEL(z, n, data) \
	std::vector<ChannelRead<BOOST_PP_CAT(type,n)>> BOOST_PP_CAT(c,n);\

// create:
// foreach(auto& channel, mChannel(n)){
//   c(n).push_back(channel.read(ic.getTimestamp(), mTolerance));
// }
#define MIRA_FW_INTERNAL_PUSH_CHANNEL_READ(z, n, data) \
	foreach(auto& channel, BOOST_PP_CAT(mChannel,n)){ \
		BOOST_PP_CAT(c,n).push_back(channel.read(ic.getTimestamp(), mTolerance)); \
	}; \

// create:
// typename std::vector<ChannelRead<type0>>::const_iterator cIter(n) = c(n).begin();
// std::vector<Time>::iterator lastIter(n) = mLast(n).begin();
// for(; cIter(n) != c(n).end(); ++cIter(n), ++lastIter(n)){
//   if(*lastIter(n) == cIter(n)->getTimestamp())
//		return;
// }
#define MIRA_FW_INTERNAL_CALL_RET(z, n, data) \
    typename std::vector<ChannelRead<BOOST_PP_CAT(type,n)>>::const_iterator BOOST_PP_CAT(cIter,n) = BOOST_PP_CAT(c,n).begin(); \
			  std::vector<Time>::iterator BOOST_PP_CAT(lastIter,n) = BOOST_PP_CAT(mLast,n).begin();\
	for(; BOOST_PP_CAT(cIter,n) != BOOST_PP_CAT(c,n).end(); ++BOOST_PP_CAT(cIter,n), ++BOOST_PP_CAT(lastIter,n)){ \
		if(*BOOST_PP_CAT(lastIter,n) == BOOST_PP_CAT(cIter,n)->getTimestamp()) \
			return; \
	}

// create:
// mChannelReadn = c(n);
#define MIRA_FW_INTERNAL_GET_CHANNELREAD(z, n, data) \
	BOOST_PP_CAT(mChannelRead,n) = BOOST_PP_CAT(c,n);

// create:
// void callback0(ChannelRead<type0> ic) {
//   std::vector<ChannelRead<type0>> c0;
//   std::vector<ChannelRead<type1>> c1;
//   ...
//
//   try{
//	   for(auto& channel:mChannel0){
//       c0.push_back(channel.read(ic.getTimestamp(), mTolerance));
//	   };
//	   for(auto& channel:mChannel1){
//       c1.push_back(channel.read(ic.getTimestamp(), mTolerance));
//	   };
//     ...
//   }
//   catch(...)
//   { return; }
//   /* now we can call the function with all channel elements */
//   call( c0, c1, ...);
// }
#define MIRA_FW_INTERNAL_GENCALLBACK(z, n, data) \
	void BOOST_PP_CAT(callback,n)(ChannelRead<BOOST_PP_CAT(type,n)> ic) { \
		BOOST_PP_REPEAT(data,MIRA_FW_INTERNAL_DECL_CHANNEL,) \
		try{ \
			BOOST_PP_REPEAT(data,MIRA_FW_INTERNAL_PUSH_CHANNEL_READ,) \
		}\
		catch(...)\
		{ return; }\
		/* now we can call the function with all channel elements */\
		call(BOOST_PP_REPEAT(data,MIRA_FW_INTERNAL_NUMBER,c));\
	}\

// create:
// cItern = cn.begin();
// lastItern = mLastn.begin();
// for(; cItern != cn.end(); ++cItern, ++lastItern){
//   *lastItern = cItern->getTimestamp();
// }
#define MIRA_FW_INTERNAL_CALL_ASSIGN(z, n, data) \
    BOOST_PP_CAT(cIter,n) = BOOST_PP_CAT(c,n).begin(); \
    BOOST_PP_CAT(lastIter,n) = BOOST_PP_CAT(mLast,n).begin();\
	for(; BOOST_PP_CAT(cIter,n) != BOOST_PP_CAT(c,n).end(); ++BOOST_PP_CAT(cIter,n), ++BOOST_PP_CAT(lastIter,n)){ \
		*BOOST_PP_CAT(lastIter,n) = BOOST_PP_CAT(cIter,n)->getTimestamp(); \
	} \

// create:
// foreach(auto& channel, mChannel(n)){
//   timeStamp = channel.read().getTimestamp();
//   if ( timeStamp < oldestTimeStamp )
//     oldestTimeStamp = timeStamp;
// }
#define MIRA_FW_INTERNAL_OLDESTTIMESTAMP(z, n, data) \
	foreach(auto& channel, BOOST_PP_CAT(mChannel,n)){ \
		timeStamp = channel.read().getTimestamp(); \
		if ( timeStamp < oldestTimeStamp ) \
			oldestTimeStamp = timeStamp; \
	} \

// create:
// foreach(auto& channel, mChannel(n)){
//   cn.push_back(channel.read(oldestTimeStamp, mTolerance));
// };
#define MIRA_FW_INTERNAL_READ_FROM_OLD(z, n, data) \
	foreach(auto& channel, BOOST_PP_CAT(mChannel,n)){ \
		BOOST_PP_CAT(c,n).push_back(channel.read(oldestTimeStamp, mTolerance)); \
	};\


// data contains 2 tuple elements 1: number of template arguments, 2: Class name
#define MIRA_FW_INTERNAL_SUBSCRIBE(z, n, data) \
	foreach(const auto& channelID, (BOOST_PP_CAT(channelID,n))) \
    { \
		(BOOST_PP_CAT(mLast,n)).push_back(Time::unixEpoch()); \
		(BOOST_PP_CAT(mChannelRead,n)).push_back(ChannelRead<BOOST_PP_CAT(type,n)>()); \
		(BOOST_PP_CAT(mChannel,n)).push_back(authority.subscribe<BOOST_PP_CAT(type,n)>(channelID, &BOOST_PP_TUPLE_ELEM(2,1,data)<\
				BOOST_PP_REPEAT(BOOST_PP_TUPLE_ELEM(2,0,data),MIRA_FW_INTERNAL_NUMBER,type)>::BOOST_PP_CAT(callback,n), this, t)); \
	}

// create:
// while(mChannel(n).size() != 0){
//	 authority.unsubscribe<typen>(mChannel(n).back().getID());
//	 mLast(n).pop_back();
//	 mChannelRead(n).pop_back();
//	 mChannel(n).pop_back();
// }
#define MIRA_FW_INTERNAL_UNSUBSCRIBE(z, n, data) \
	while(BOOST_PP_CAT(mChannel,n).size() != 0){ \
		authority.unsubscribe<BOOST_PP_CAT(type,n)>(BOOST_PP_CAT(mChannel,n).back().getID()); \
		BOOST_PP_CAT(mLast,n).pop_back(); \
		BOOST_PP_CAT(mChannelRead,n).pop_back(); \
		BOOST_PP_CAT(mChannel,n).pop_back(); \
	} \


// Use this macro to generate a synchronizer class with the given name
// and the given number of channel types
#define MIRA_CHANNEL_SYNCHRONIZER( TNAME, TNUM ) \
	template < \
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,typename type) \
		> class TNAME\
	{\
	public:\
	\
	/* generate constructor which is empty*/ \
	TNAME() \
	{} \
	\
	void unsubscribe(Authority& authority)\
	{\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_UNSUBSCRIBE,(TNUM,TNAME))\
	}\
	\
	void subscribe(Authority& authority,\
			BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,const std::vector<std::string>& channelID)\
			\
			,const Duration& t = Duration::seconds(0))\
	{\
		mFunction = NULL;\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_SUBSCRIBE,(TNUM,TNAME))\
		mTolerance = t;\
	}\
	\
	void subscribe(Authority& authority,\
			BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,const std::vector<std::string>& channelID)\
			\
			,boost::function<void (BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,))> fn\
			,const Duration& t = Duration::seconds(0))\
	{\
		mFunction = fn;\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_SUBSCRIBE,(TNUM,TNAME))\
		mTolerance = t;\
	}\
	\
	/* generate subscribe method provided for convenience*/\
	template<typename Class>\
	void subscribe(Authority& authority,\
			BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,const std::vector<std::string>& channelID)\
			\
			, void (Class::*f)( BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,) )\
			, Class* obj\
			, const Duration& t = Duration::seconds(0)\
			)\
	{\
		subscribe(authority,\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,channelID),\
				boost::bind(f, obj, BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_INC_NUMBER,_)),t);\
	}\
	/* generate callback functions for every channel*/ \
	BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_GENCALLBACK,TNUM)\
	\
	void call(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL_VAR,))\
	{\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CALL_RET,)\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CALL_ASSIGN,)\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_GET_CHANNELREAD,)\
		if (mFunction) mFunction( BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,c));\
	}\
	\
	boost::tuple<BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,)> read() { \
		if(!isValid()) \
			MIRA_THROW(XInvalidRead, "Cannot read from synchronized channel "\
					    << "' since there is no data");\
		return boost::make_tuple(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,mChannelRead));\
	}\
	\
	bool isValid() const { \
		/*bool allValid = true;*/ \
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_ISVALID,mChannelRead) \
		/*return allValid;*/ \
		return true; \
	}\
	\
	boost::tuple<BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,)> waitForData(const Duration& timeout = Duration::infinity()) \
	{\
		/* validateReadAccess is private :( */ \
		/*BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_VALIDATE,mChannel)*/\
		\
		Time end;\
		if(!timeout.isValid() || timeout.isInfinity())\
			end = Time::eternity();\
		else\
			end = Time::now() + timeout;\
		\
		while(!boost::this_thread::interruption_requested())\
		{\
			Time oldestTimeStamp = Time::now();\
			Time timeStamp;\
			try {\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_DECL_CHANNEL,)\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_OLDESTTIMESTAMP,)\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_READ_FROM_OLD,)\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_GET_CHANNELREAD,)\
				return read();\
			} catch(XInvalidRead& ex) {}\
			if(Time::now()>end) /* handle timeout */\
				break;\
			MIRA_SLEEP(50);\
		}\
		return boost::make_tuple(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,mChannelRead));\
	}\
	\
	bool waitForPublisher(const Duration& timeout = Duration::infinity()) const\
	{\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER,) \
		return true; \
	}\
	\
	BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_MEMBER,)\
	Duration mTolerance; \
	boost::function<void (BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,))> mFunction; \
	};\
	\


#define MIRA_FW_GEN_CHANNEL_SYNCHRONIZER( z, n, data) \
	MIRA_CHANNEL_SYNCHRONIZER(BOOST_PP_CAT(ChannelVectorSynchronizer,n), n)\
	/* NEXT SYNCHRONIZER */\

// generate predefined channel synchronizer for 1 vector of channels
// MIRA_CHANNEL_SYNCHRONIZER(BOOST_PP_CAT(ChannelVectorSynchronizer, 1), 1)
// generate predefined channel synchronizer for 1 to 9 vectors of channels
BOOST_PP_REPEAT_FROM_TO(1,10,MIRA_FW_GEN_CHANNEL_SYNCHRONIZER,)

#undef MIRA_FW_INTERNAL_NUMBER
#undef MIRA_FW_INTERNAL_INC_NUMBER
#undef MIRA_FW_INTERNAL_CHANNEL
#undef MIRA_FW_INTERNAL_CHANNEL_VAR
#undef MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER
#undef MIRA_FW_INTERNAL_ISVALID
#undef MIRA_FW_INTERNAL_MEMBER
#undef MIRA_FW_INTERNAL_DECL_CHANNEL
#undef MIRA_FW_INTERNAL_PUSH_CHANNEL_READ
#undef MIRA_FW_INTERNAL_CALL_RET
#undef MIRA_FW_INTERNAL_GET_CHANNELREAD
#undef MIRA_FW_INTERNAL_GENCALLBACK
#undef MIRA_FW_INTERNAL_CALL_ASSIGN
#undef MIRA_FW_INTERNAL_OLDESTTIMESTAMP
#undef MIRA_FW_INTERNAL_READ_FROM_OLD
#undef MIRA_FW_INTERNAL_SUBSCRIBE
#undef MIRA_FW_INTERNAL_UNSUBSCRIBE
#undef MIRA_CHANNEL_SYNCHRONIZER
#undef MIRA_FW_GEN_CHANNEL_SYNCHRONIZER

#endif

///////////////////////////////////////////////////////////////////////////////

}

#endif
