/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RecoverTest.C
 *    .
 *
 * @author Tim Langner
 * @date   2012/01/17
 */

#include <fw/Unit.h>

namespace mira { namespace fw {

///////////////////////////////////////////////////////////////////////////////

/// Example unit of how to use recovering in Units
class RecoverTestPublisherUnit : public Unit
{
	/// Register at class factory
	MIRA_OBJECT(RecoverTestPublisherUnit)
public:

	/// Tell base class to call process in a 500 millisecond interval
	RecoverTestPublisherUnit() :
		Unit(Duration::milliseconds(500))
	{
		mCounter = 0;
	}

	/// Overwritten from MicroUnit. Gets called once on startup
	virtual void initialize()
	{
		// publish int channel and store it in a member
		mIntChannel = publish<int>("Int");
	}

	/// Overwritten from Unit. Gets called every 500 milliseconds
	virtual void process(const Timer& timer)
	{
		if (mCounter == 10)
		{
			// Signal that we are in an error state and need recovery
			// from now on recover will be called instead of process
			needRecovery("We reached 10. This is a bad number. Try to recover from that");
			return;
		}
		// get write proxy object from int channel
		ChannelWrite<int> value = mIntChannel.write();
		// and write our incremented counter to it
		value->value() = mCounter++;
		value->timestamp = timer.current;
	}

	/// Overwritten from MicroUnit. Gets called instead of process in recovery mode
	/// every mRecoverInterval (default 1 second)
	void recover()
	{
		mCounter = 11;
		sleep(Duration::seconds(5));
		// signal that we are operating normally again - process will be called again
		operational();
	}

protected:

	Channel<int> mIntChannel;
	int mCounter;
};

///////////////////////////////////////////////////////////////////////////////

/// Example unit of how to use recovering in MicroUnits
class RecoverTestSubscriberUnit : public MicroUnit
{
	/// Register at class factory
	MIRA_OBJECT(RecoverTestSubscriberUnit)
public:

	/// Overwritten from MicroUnit. Gets called once on startup
	virtual void initialize()
	{
		// subscribe to int channel with callback
		subscribe<int>("Int", &RecoverTestSubscriberUnit::myCallback);
	}

	/// Gets called every time new data is available in int channel
	void myCallback(ChannelRead<int> value)
	{
		MIRA_LOG(NOTICE) << "RecoverTestSubscriberUnit: Got callback. The value is " << value->value();
		if (value->value() == 20)
			// Signal that we are in an error state and need recovery 
			needRecovery("Reached 20. Something failed so we need to recover");
	}

	/// Overwritten from MicroUnit. Gets called in recovery mode
	/// every mRecoverInterval (default 1 second)
	virtual void recover()
	{
		sleep(Duration::seconds(5));
		// signal that we are operating normally again
		operational();
	}
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::fw::RecoverTestPublisherUnit, mira::Unit);
MIRA_CLASS_SERIALIZATION(mira::fw::RecoverTestSubscriberUnit, mira::MicroUnit);
