/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TestXMLMacrosErrors.C
 *    Tests for macro expansion error messages.
 *
 * @author Adrian Kriegel
 * @date   Wed Feb 12 2025
 */

#include <boost/format.hpp>
#include <boost/test/unit_test.hpp>

#include <xml/XMLDom.h>
#include <xml/XMLDomPreprocessor.h>
#include <error/Exceptions.h>
#include <utils/PackageFinder.h>
#include <xml/macros/Types.h>

#include <xml/macros/TestUtils.h>

using namespace mira;
using namespace mira::xmlmacros;

inline Exception getError(const std::string& xml)
{
	XMLDom dom;
	dom.loadFromString(xml);
	dom.setUri("path/to/file.xml");
	try {
		XMLDomPreprocessor().preprocessXML(dom);
	}
	catch (Exception e) {
		return e;
	}

	MIRA_THROW(XAssertion, "Failed to throw.")
}

/** @returns trimmed lines of @param s that are not just whitespace. */
inline std::vector<std::string> splitAndTrim(const std::string& s)
{
	std::vector<std::string> result;
	std::istringstream ss(s);
	for (std::string line; std::getline(ss, line);) {
		boost::trim(line);
		if (!line.empty())
			result.push_back(line);
	}

	return result;
}

/**
 * Checks that each line in the error message starts with the corresponding line in @param expected.
 */
inline void checkError(const Exception& e, const std::string& expected)
{
	const std::string actual = e.what();

	auto expectedLines = splitAndTrim(expected);
	auto actualLines = splitAndTrim(actual);

	BOOST_CHECK_EQUAL(expectedLines.size(), actualLines.size());

	for (uint i = 0; i < expectedLines.size(); ++i) {
		BOOST_CHECK_EQUAL(expectedLines[i], actualLines.at(i).substr(0, expectedLines[i].size()));
	}
}

BOOST_AUTO_TEST_CASE(TestMacroErrorMissingParameter)
{
	{
		const std::string config = R""(
            <macro:Define_Foo>
                <Parameters>
                    <MyParam />
                </Parameters>
                <Body></Body>
            </macro:Define_Foo>

            <macro:Define_MacroWithError>
                <Body>
                    <!-- Missing parameter -->
                    <macro:Foo />
                </Body>
            </macro:Define_MacroWithError>

            <macro:MacroWithError />
        )"";

		auto error = getError(wrapInRoot(config));
        
        checkError(error, R"(
            Macro parameter "MyParam" is required but missing.
            while deserializing 'Foo'
            in tag starting at: path/to/file.xml(12)
            expanding <macro:Foo> at path/to/file.xml(12) defined at path/to/file.xml(2)
            expanding <macro:MacroWithError> at path/to/file.xml(16) defined at path/to/file.xml(9)
        )");
	}


    {
		const std::string config = R""(
            <root
                xmlns:macro="http://www.mira-project.org/xmlns:macro"
                xmlns:mymodule="MIRABase:tests/xml/macros/etc/ImportedMacro.xml"
            >
                <macro:Define_MacroWithError>
                    <Body>
                        <!-- Missing parameter -->
                        <mymodule:MyImportedMacro />
                    </Body>
                </macro:Define_MacroWithError>

                <macro:MacroWithError />
            </root>
        )"";

		auto error = getError(config);
        
        const auto expected = boost::format(R"(
            Macro parameter "MyParam" is required but missing.
            while deserializing 'MyImportedMacro'
            in tag starting at: path/to/file.xml(9)
            expanding <mymodule:MyImportedMacro> at path/to/file.xml(9) defined at %1%/tests/xml/macros/etc/ImportedMacro.xml(2)
            expanding <macro:MacroWithError> at path/to/file.xml(13) defined at path/to/file.xml(6)
        )") % findPackage("MIRABase").string();

		checkError(error, expected.str());
	}
}

BOOST_AUTO_TEST_CASE(TestMaxDepth)
{
    const std::string config = R""(
        <macro:Define_Foo>
            <Body>
                <macro:Foo />
            </Body>
        </macro:Define_Foo>

        <macro:Foo />
    )"";

    XMLDom dom;
    dom.loadFromString(wrapInRoot(config));

    BOOST_CHECK_THROW(XMLDomPreprocessor().preprocessXML(dom), XMaxDepth);
}
