/*
 * Copyright (C) 2013 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SingletonThreadingTest.C
 *    Implementation of SingletonThreadingTest.h.
 *
 * @author Erik Einhorn
 * @date   13/03/21
 */

#include <boost/test/unit_test.hpp>

#include <utils/Singleton.h>
#include <utils/Time.h>

#include <thread/Atomic.h>
#include <thread/Thread.h>

using namespace mira;

class LazySingletonUnlocked : public Singleton<LazySingletonUnlocked,
                        singleton::LazyInstantiation,singleton::CreateUsingNew,
                        singleton::NormalLifetime,singleton::NoLock>
{
public:
	LazySingletonUnlocked() : a(0) {
		std::cout << "LazySingletonUnlocked" << std::endl;
		atomic::inc(&count);
		MIRA_SLEEP(100); // wait a little to make sure we have a concurrent access
	}

	int a;

	static void test() {
		LazySingletonUnlocked::instance().a = 1;
	}

	static volatile uint32 count;
};
volatile uint32 LazySingletonUnlocked::count = 0;


class LazySingletonLocked : public Singleton<LazySingletonLocked,
                        singleton::LazyInstantiation,singleton::CreateUsingNew,
                        singleton::NormalLifetime,singleton::MutexLock>
{
public:
	LazySingletonLocked() : a(0) {
		std::cout << "LazySingletonLocked" << std::endl;
		atomic::inc(&count);
		MIRA_SLEEP(100); // wait a little to make sure we have a concurrent access
	}

	int a;

	static void test() {
		LazySingletonLocked::instance().a = 1;
	}

	static volatile uint32 count;
};
volatile uint32 LazySingletonLocked::count = 0;

// tests mutex protected creation with concurrent access
BOOST_AUTO_TEST_CASE( testCreation )
{
	// LazySingletonUnlocked is not protected against concurrent access
	// and therefore will be instantiated twice, which is a problem
	boost::thread t1(&LazySingletonUnlocked::test);
	boost::thread t2(&LazySingletonUnlocked::test);
	t1.join();
	t2.join();
	BOOST_CHECK_EQUAL(LazySingletonUnlocked::count, 2);


	// LazySingletonLocked IS protected ...
	boost::thread t3(&LazySingletonLocked::test);
	boost::thread t4(&LazySingletonLocked::test);
	t3.join();
	t4.join();
	// ... and should be instantiated only ONCE!
	BOOST_CHECK_EQUAL(LazySingletonLocked::count, 1);
}

// tests mutex protected creation with concurrent access
BOOST_AUTO_TEST_CASE( testPerformance )
{
	const int N = 10000000;

	Time start = Time::now();
	for(int i=0; i<N; ++i)
		LazySingletonUnlocked::instance().a++;
	Duration diff =  Time::now()-start;
	std::cout << "Duration with LazySingletonUnlocked: " << diff << std::endl;

	start = Time::now();
	for(int i=0; i<N; ++i)
		LazySingletonLocked::instance().a++;
	diff =  Time::now()-start;
	std::cout << "Duration with LazySingletonLocked: " << diff<< std::endl;


	std::cout << LazySingletonUnlocked::instance().a << std::endl; // prevent compiler optimization
	std::cout << LazySingletonLocked::instance().a << std::endl; // prevent compiler optimization
}
