/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationDesireVersionTest.C
 *    Test for serialization framework desireClassVersions().
 *
 * @author Christof Schröter
 * @date   2021/10/13
 */

#ifndef Q_MOC_RUN
#include <boost/test/unit_test.hpp>
#endif

#include <error/Logging.h>
#include <error/LogConsoleSink.h>

#include <serialization/BinaryJSONConverter.h>
#include <serialization/MetaSerializer.h>

#include "CommonTest.h"

///////////////////////////////////////////////////////////////////////////////

class ClassLegacy {
public:

	ClassLegacy() : m(0) {}

	void check() {
		BOOST_CHECK_EQUAL(m, 10);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("m", m, "");
	}

protected:
	int m;
};

class ClassNotAcceptingDesiredVersion {
public:
	ClassNotAcceptingDesiredVersion() : m(0), n(0) {}
	ClassNotAcceptingDesiredVersion(bool) : m(10), n(20) {}

	void check() {
		BOOST_CHECK_EQUAL(m, 10);
		BOOST_CHECK_EQUAL(n, 20);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		// this actually WOULD handle a desired version (but it does not declare that to the reflector)
		serialization::VersionType version = r.version(2, this);
		r.member("m", m, "");
		if (version >= 2)
			r.member("n", n, "");
	}

	// the more realistic case would be this:
//	void reflect(BinaryStreamSerializer& r)
//	{
//		r.version(2, this);
//		r.member("m", m, "");
//		r.member("n", n, "");
//	}

protected:
	int m, n;
};

class ClassAcceptingDesiredVersion {
public:
	ClassAcceptingDesiredVersion() : m(0), n(0) {}
	ClassAcceptingDesiredVersion(bool) : m(10), n(20) {}

	void check() {
		BOOST_CHECK_EQUAL(m, 10);
		BOOST_CHECK_EQUAL(n, 20);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType version = r.version(2, serialization::AcceptDesiredVersion(), this);
		r.member("m", m, "");
		if (version >= 2)
			r.member("n", n, "");
	}

protected:
	int m, n;
};

///////////////////////////////////////////////////////////////////////////////

template<typename Class>
void testBinaryStream(serialization::VersionType desiredVersion)
{
	// serialize test class
	Class c(true);
	std::stringbuf buffer;
	BinaryStreamSerializer s(&buffer);

	serialization::ClassVersionMap desiredVersions;
	desiredVersions[typeName<Class>()] = desiredVersion;
	s.desireClassVersions(desiredVersions);

	s.serialize(c, false); // false = ignore the class name mismatch!
	//cout << ">" << print(buffer) << "<" << endl;

	// deserialize legacy class
	ClassLegacy c2; // with 0

	BinaryStreamDeserializer s2(&buffer);
	s2.deserialize(c2, false);
	c2.check();
}

void testMeta(serialization::VersionType desiredVersion)
{
	serialization::ClassVersionMap desiredVersions;
	desiredVersions[typeName<ClassAcceptingDesiredVersion>()] = desiredVersion;

	ClassAcceptingDesiredVersion c(true);

	// serialize to JSON
	JSONSerializer j;
	j.desireClassVersions(desiredVersions);

	JSONValue val = j.serialize(c);
//	std::cout << val << std::endl;

	// serialize to binary buffer
	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializer s(&buffer);
	s.desireClassVersions(desiredVersions);

	s.serialize(c);

	// create meta
	MetaTypeDatabase db;
	MetaSerializer ms(db);
	ms.desireClassVersions(desiredVersions);

	auto meta = ms.addMeta(c);

	// create JSON from binary + meta
	JSONValue valFromBin;
	BinaryJSONConverter::binaryToJSON(buffer, true, *meta, db, valFromBin);
//	std::cout << valFromBin << std::endl;

	BOOST_CHECK_EQUAL(json::write(val), json::write(valFromBin));
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( BasicSerializationTest )
{
	// basic test that serialization for ClassNotAcceptingDesiredVersion/ClassAcceptingDesiredVersion
	// works as expected (without desiring a version)

	testAll<ClassNotAcceptingDesiredVersion>("ClassNotAcceptingDesiredVersion",1);
	testAll<ClassAcceptingDesiredVersion>("ClassAcceptingDesiredVersion",1);
}

BOOST_AUTO_TEST_CASE( DesireVersionTest )
{
	// test that desiring a version works as expected for binary serialization,
	// by trying to deserialize to legacy class (test both compatible and conflicting cases)

	LogConsoleSink sink;
	MIRA_LOGGER.registerSink(sink);

	// serialize current class and deserialize to legacy class

	// "Tried to deserialize versioned binary data (version 2) into unversioned type ..."
	BOOST_CHECK_THROW(testBinaryStream<ClassNotAcceptingDesiredVersion>(2), XIO);

	// Same error as above (the desired version 0 is ignored, a warning is shown)
	BOOST_CHECK_THROW(testBinaryStream<ClassNotAcceptingDesiredVersion>(0), XIO);

	// Same error as above
	BOOST_CHECK_THROW(testBinaryStream<ClassAcceptingDesiredVersion>(2), XIO);

	// works by desriring version 0 for serialization
	testBinaryStream<ClassAcceptingDesiredVersion>(0);
}

BOOST_AUTO_TEST_CASE( DesireVersionMetaTest )
{
	// test that desiring a version works with JSONSerializer and MetaSerializer
	// (test Binary+Meta == JSON with current and desired version)

	testMeta(2);
	testMeta(0);
}
