/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationCompatibilityForceV0Test.C
 *    This test needs to run in a separate process, as the check for
 *    the environment variable is done only once per process
 *    (for performance reasons)
 *
 * @author Christof Schröter
 * @date   2019/06/25
 */

#include <boost/test/unit_test.hpp>

#include <boost/algorithm/string/split.hpp>
#include <boost/range/algorithm_ext/erase.hpp>

#include <serialization/BinarySerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/XMLSerializer.h>

using namespace mira;

struct MimicBinaryV2TestClass
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("int16", i16, "");
		r.member("int8",  i8, "");
		r.member("int",   i, "");
	}

	bool operator==(const MimicBinaryV2TestClass& other) const {
		return (i16 == other.i16) && (i8 == other.i8) && (i == other.i);
	}

	uint16 i16;
	uint8  i8;
	int    i;
};

// also see SerializationCompatibilityTest.C, test case DeserializeV0MimicV2Test
// for resulting behaviour WITHOUT the environment variable
BOOST_AUTO_TEST_CASE(ForceBinaryDeserializeV0Test)
{
#ifdef CHECK_FORCE_SERIALIZE_BINARY_VERSION  // this only works in debug mode
	// force deserialization as v0
	BOOST_CHECK_EQUAL(setenv("MIRA_FORCE_DESERIALIZE_VERSION", "0", 0), 0);

	// write v0 binary format, but mimic v2!
	MimicBinaryV2TestClass c;
	c.i16 = BINARY_VERSION_MARKER;
	c.i8 = BinaryBufferSerializer::getSerializerFormatVersion();
	c.i = 12345;

	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializerLegacy s(&buffer);
	s.serialize(c, false);

	// now the deserializer will not mistake the format for v2
	BinaryBufferDeserializer d(&buffer);
	MimicBinaryV2TestClass c2;
	d.deserialize(c2, false);

	BOOST_CHECK(c == c2);
#endif
}

// also see SerializationCompatibilityTest.C, test case SerializeReadV0Test
// for resulting behaviour WITHOUT the environment variable
BOOST_AUTO_TEST_CASE(ForceBinarySerializeV0Test)
{
#ifdef CHECK_FORCE_SERIALIZE_BINARY_VERSION  // this only works in debug mode
	// force serialization as v0
	BOOST_CHECK_EQUAL(setenv("MIRA_FORCE_SERIALIZE_VERSION", "0", 0), 0);

	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializer s(&buffer);

	int i = 12345;
	s.serialize(i, false);

	i = 54321;
	s.serialize(i, false);

	// now reading with legacy deserializer will work
	BinaryBufferDeserializerLegacy d(&buffer);

	d.deserialize(i, false);
	BOOST_CHECK_EQUAL(i, 12345);

	d.deserialize(i, false);
	BOOST_CHECK_EQUAL(i, 54321);
#endif
}

struct VersionTestClass
{
	VersionTestClass() : i(4321) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.version(2, this);
		r.member("int",   i, "");
	}

	int i;
};

std::list<std::string> refxml = {"<root>",
                                 "  <!--Test-->",
                                 "  <test version=\"2\">",
                                 "    <int>4321</int>",
                                 "  </test>",
                                 "</root>"
                                };

std::string refjson = "{\"@version\":2,\"int\":4321}";

BOOST_AUTO_TEST_CASE(ForceXMLSerializeV0Test)
{
	// force serialization as v0
	BOOST_CHECK_EQUAL(setenv("MIRA_FORCE_SERIALIZE_VERSION", "0", 0), 0);

	VersionTestClass c;

	XMLDom xml;
	XMLSerializer serializer(xml);
	serializer.serialize("test", c, "Test");
	std::string s = xml.saveToString();

	// remove first line (xml version & encoding)
	std::list<std::string> lines;
	boost::split(lines, s, boost::is_any_of("\n"));
	lines.pop_front();

	 // remove formatting (newlines and whitespace)
	s = boost::algorithm::join(lines, "");
	boost::remove_erase_if(s, boost::is_any_of(" \t"));

	std::string ref = boost::algorithm::join(refxml, "");
	boost::remove_erase_if(ref, boost::is_any_of(" \t"));

	BOOST_CHECK_EQUAL(s, ref);
}

BOOST_AUTO_TEST_CASE(ForceJSONSerializeV0Test)
{
	// force serialization as v0
	BOOST_CHECK_EQUAL(setenv("MIRA_FORCE_SERIALIZE_VERSION", "0", 0), 0);

	VersionTestClass c;

	XMLDom xml;
	JSONSerializer serializer;
	json::Value json = serializer.serialize(c);
	std::string s = json::write(json);

	BOOST_CHECK_EQUAL(s, refjson);
}

