/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TypenameTest.C
 *    Typename test cases.
 *
 * @author Tim Langner
 * @date   2010/10/20
 */

#include <vector>
#include <map>

#include <boost/test/unit_test.hpp>

#include <platform/Types.h>
#include <platform/Typename.h>

#include <transform/RigidTransform.h>

#include <iostream>

using namespace mira;

template <typename>
struct MyStruct {};

namespace {

struct StructInAnonymousNamespace {};

} // namespace

enum MyEnum
{
	VALUE1 = 0
};

struct Elongation {};

template <typename T, bool checkTypenameWithTArgument = std::is_default_constructible<T>::value>
struct checkTypeName {
	checkTypeName(const std::string& name) {
		BOOST_CHECK_EQUAL(typeName<T>(), name);
		T t{};
		BOOST_CHECK_EQUAL(typeName(t), name);
	}
};

template <typename T>
struct checkTypeName<T, false> {      // specialization for non-default-constructible T
	checkTypeName(const std::string& name) {
		BOOST_CHECK_EQUAL(typeName<T>(), name);
	}
};

BOOST_AUTO_TEST_CASE( TestTypename )
{
	// test basic types
	checkTypeName<char>("char");
	checkTypeName<uint8>("unsigned char");
	checkTypeName<uint16>("unsigned short");
	checkTypeName<uint32>("unsigned int");
	checkTypeName<uint64>("unsigned int64");
	checkTypeName<int8>("signed char");
	checkTypeName<int16>("short");
	checkTypeName<int32>("int");
	checkTypeName<int64>("int64");
	checkTypeName<float>("float");
	checkTypeName<double>("double");
	checkTypeName<long double>("double128");
	checkTypeName<std::string>("std::string");
	checkTypeName<std::out_of_range>("std::out_of_range");

	checkTypeName<MyEnum>("MyEnum");

	// test reference and pointers
	checkTypeName<int64&>("int64");
	typedef int const* const ConstIntConstPtr;
	checkTypeName<ConstIntConstPtr>("int const*const");
	checkTypeName<MyStruct<ConstIntConstPtr>>("MyStruct<int const*const>");
	checkTypeName<ConstIntConstPtr*>("int const*const*");
	checkTypeName<const MyStruct<ConstIntConstPtr>*>("MyStruct<int const*const>const*");

	// test containers
	checkTypeName<std::vector<int> >("std::vector<int,std::allocator<int>>");
	checkTypeName<std::vector<const int*> >("std::vector<int const*,std::allocator<int const*>>");
	typedef std::map<std::string, RigidTransform2f> PoseMap;
	checkTypeName<PoseMap>("std::map<std::string,mira::RigidTransform<float,2>,"
	                       "std::less<std::string>,"
	                       "std::allocator<std::pair<std::string const,"
	                       "mira::RigidTransform<float,2>>>>");

	typedef std::map<int, Eigen::Matrix4f, std::less<int>,
	                 Eigen::aligned_allocator<Eigen::Matrix4f> > Matrix4Map;
	checkTypeName<Matrix4Map>("std::map<int,Eigen::Matrix<float,4,4,0,4,4>,"
	                          "std::less<int>,"
	                          "Eigen::aligned_allocator<Eigen::Matrix<float,4,4,0,4,4>>>");

	// test special cases like anonymous namespaces and volatile
	checkTypeName<StructInAnonymousNamespace>("(anonymous namespace)::StructInAnonymousNamespace");
	checkTypeName<std::list<StructInAnonymousNamespace> >("std::list<(anonymous namespace)::StructInAnonymousNamespace,"
	                                                      "std::allocator<(anonymous namespace)::StructInAnonymousNamespace>>");
	checkTypeName<MyStruct<volatile int> >("MyStruct<int volatile>");

	checkTypeName<Elongation>("Elongation");
}

