/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file InterpolatorTest.C
 *    Test for interpolators.
 *
 * @author Tim van der Grinten
 * @date   2021/05/26
 */


#include <boost/test/unit_test.hpp>
#if BOOST_VERSION >= 107100
#include <boost/test/tools/floating_point_comparison.hpp>
#else
#include <boost/test/floating_point_comparison.hpp>
#endif

#include <error/Logging.h>
#include <error/Exceptions.h>

#include <math/NearestNeighborInterpolator.h>
#include <math/LinearInterpolator.h>
#include <math/CRSplineInterpolator.h>

#include <iostream>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( InterpolatorTest1D )
{
	// support points
	float x1 = -0.75f, y1 = -1.0f;
	float x2 = -0.25f, y2 = -1.0f;
	float x3 = 0.25f, y3 = 1.0f;
	float x4 = 0.75f, y4 = 1.0f;

	// support ranges
	std::vector<float> xs1{x1, x2};
	std::vector<float> ys1{y1, y2};

	std::vector<float> xs2{x2, x3};
	std::vector<float> ys2{y2, y3};

	std::vector<float> xs3{x3, x4};
	std::vector<float> ys3{y3, y4};

	std::vector<float> xs{x1, x2, x3, x4};
	std::vector<float> ys{y1, y2, y3, y4};

	// test points
	std::vector<float> xx = {-1.0f, -0.9f, -0.8f, -0.7f, -0.6f, -0.5f, -0.4f, -0.3f, -0.2f, -0.1f, 0.0f, 0.1f, 0.2f, 0.3f, 0.4f, 0.5f, 0.6f, 0.7f, 0.8f, 0.9f, 1.0f};

	// ground truth data for Nearest Neighbor (nn), Linear (li) and CRSpline (cr)
	std::vector<float> nn = {-1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f};
	std::vector<float> li = {-1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -0.8f, -0.4f, 0.0f, 0.4f, 0.8f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f};
	std::vector<float> cr = {11.000f, 7.164f, 4.192, 1.988f, 0.456f, -0.5f, -0.976f, -1.068f, -0.872f, -0.484f, 0.0f, 0.484f, 0.872f, 1.068f, 0.976f, 0.5f, -0.456f, -1.988f, -4.192f, -7.164f, -11.000f};

	for (unsigned int i = 0; i < xx.size(); ++i) {
		// test point
		float x = xx[i];

		// select range to process in
		std::vector<float>& xs = (x < x2 ? xs1 : (x < x3 ? xs2 : xs3));
		std::vector<float>& ys = (x < x2 ? ys1 : (x < x3 ? ys2 : ys3));

		// apply interpolator to test point in selected range
		float y = NearestNeighborInterpolator().apply<float, float>(xs, ys, x);

		// check result
		BOOST_CHECK(y == nn[i]);
	}

	for (unsigned int i = 0; i < xx.size(); ++i) {
		// test point
		float x = xx[i];

		// select range to process in
		std::vector<float>& xs = (x < x2 ? xs1 : (x < x3 ? xs2 : xs3));
		std::vector<float>& ys = (x < x2 ? ys1 : (x < x3 ? ys2 : ys3));

		// apply interpolator to test point in selected range
		float y = LinearInterpolator().apply<float, float>(xs, ys, x);

		// check result
		BOOST_CHECK(y == li[i]);
	}

	for (unsigned int i = 0; i < xx.size(); ++i) {
		// test point
		float x = xx[i];

		// apply interpolator to test point
		float y = CRSplineInterpolator().apply<float, float>(xs, ys, x);

		// check result
		BOOST_CHECK_CLOSE(y, cr[i], 0.001f);
	}
}
