/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LoggingTest.C
 *    An example for logging.
 *
 * @author Tim Langner
 * @date   2010/07/07
 */

#include <fstream>
#include <boost/test/unit_test.hpp>

#include <error/LogTimer.h>
#include <error/LogConsoleSink.h>
#include <error/LogCustomizableFormatter.h>
#include <error/LogCustomizableFilter.h>


using namespace mira;
using namespace std;


void aFunctionThatLogs()
{
	// Create a log timer that logs the time consumed by this function
	MIRA_LOGTIMER(CRITICAL, LogTest, "measure time of logging");

	MIRA_LOG(CRITICAL) << "Now logging different levels.";
	MIRA_LOG(CRITICAL) << "Some critical message.";
	MIRA_LOG(ERROR) << "Some error message";
	MIRA_LOG(WARNING) << "Some warning message";
	MIRA_LOG(NOTICE) << "Some notice message";
	MIRA_LOG(DEBUG) << "Some debug message";
	MIRA_LOG(TRACE) << "Some debug message";

	// Do some time consuming logging
	for ( int i=0; i<10; ++i)
		MIRA_LOG(WARNING) << "i is now " << i;
}

BOOST_AUTO_TEST_CASE(LoggingGeneralTest)
{
	// Create a customized log formatter
	LogCustomizableFormatter formatter;
	// Add some custom formats to define the log output format
	formatter << "[level="
		<< LogCustomizableFormatter::Severity() << "] at "
		<< LogCustomizableFormatter::Uptime("%f") << " micro-seconds after startup. (\""
		<< LogCustomizableFormatter::Message() << "\") in "
		<< LogCustomizableFormatter::Function();

	// Create a customized log filter
	LogCustomizableFilter filter;
	// Add a custom severity filter that logs only
	// log entries with a severity of WARNING or CRITICAL
	filter << (LogCustomizableFilter::Severity<LogCustomizableFilter::EQUAL>(WARNING) |
		LogCustomizableFilter::Severity<LogCustomizableFilter::EQUAL>(CRITICAL));

	// create a stream sink that writes to console
	LogConsoleSink txtSink;
	// and assign the custom formatter to it
	txtSink.setFormatter(formatter);

	std::ofstream file("test.log");
	// create a stream sink that writes to file and uses the default formatter
	LogTxtStreamSink fileSink(file);
	// and assign the filter to it
	fileSink.setFilter(filter);

	// register the sinks
	MIRA_LOGGER.registerSink(txtSink);
	MIRA_LOGGER.registerSink(fileSink);
	// Set the log level to TRACE (everything gets logged)
	MIRA_LOGGER.setSeverityLevel(TRACE);

	aFunctionThatLogs();
	MIRA_LOG(CRITICAL) << "Setting severity to CRITICAL";
	// Set the log level to CRITICAL (only CRITICAL log entries get logged)
	// This should boost up the execution time of aFunctionThatLogs() since
	// the log output is minimized
	MIRA_LOGGER.setSeverityLevel(CRITICAL);
	aFunctionThatLogs();
}

static uint32 numCallbacksCalled1 = 0;
static uint32 numCallbacksCalled2 = 0;
static SeverityLevel slevel1;
static SeverityLevel slevel2;

void callback1(SeverityLevel level) {
	cout << "callback1: severity level changed to " << level << endl;
	++numCallbacksCalled1;
	slevel1 = level;
}

void callback2(SeverityLevel level) {
	cout << "callback2: severity level changed to " << level << endl;
	++numCallbacksCalled2;
	slevel2 = level;
}

BOOST_AUTO_TEST_CASE(LoggingCallbackTest)
{
	MIRA_LOGGER.setSeverityLevel(NOTICE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 0);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 0);

	uint32 id1 = MIRA_LOGGER.registerCallback(&callback1);
	uint32 id2 = MIRA_LOGGER.registerCallback(&callback2);
	MIRA_LOGGER.setSeverityLevel(NOTICE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 1);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 1);

	MIRA_LOGGER.setSeverityLevel(TRACE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 2);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 2);
	BOOST_CHECK_EQUAL(slevel1, TRACE);
	BOOST_CHECK_EQUAL(slevel2, TRACE);

	MIRA_LOGGER.unregisterCallback(id2);
	MIRA_LOGGER.setSeverityLevel(NOTICE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 3);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 2);
	BOOST_CHECK_EQUAL(slevel1, NOTICE);
	BOOST_CHECK_EQUAL(slevel2, TRACE);

	MIRA_LOGGER.unregisterCallback(id1);
	MIRA_LOGGER.setSeverityLevel(DEBUG);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 3);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 2);
	BOOST_CHECK_EQUAL(slevel1, NOTICE);
	BOOST_CHECK_EQUAL(slevel2, TRACE);

	uint32 id3 = MIRA_LOGGER.registerCallback(&callback1);
	MIRA_LOGGER.setSeverityLevel(WARNING);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 4);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 2);
	BOOST_CHECK_EQUAL(slevel1, WARNING);
	BOOST_CHECK_EQUAL(slevel2, TRACE);

	MIRA_LOGGER.unregisterCallback(id3);
	MIRA_LOGGER.setSeverityLevel(NOTICE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 4);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 2);
	BOOST_CHECK_EQUAL(slevel1, WARNING);
	BOOST_CHECK_EQUAL(slevel2, TRACE);

	uint32 id4 = MIRA_LOGGER.registerCallback(&callback2);
	MIRA_LOGGER.setSeverityLevel(NOTICE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 4);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 3);
	BOOST_CHECK_EQUAL(slevel1, WARNING);
	BOOST_CHECK_EQUAL(slevel2, NOTICE);

	MIRA_LOGGER.unregisterCallback(id4);
	MIRA_LOGGER.setSeverityLevel(TRACE);
	BOOST_CHECK_EQUAL(numCallbacksCalled1, 4);
	BOOST_CHECK_EQUAL(numCallbacksCalled2, 3);
	BOOST_CHECK_EQUAL(slevel1, WARNING);
	BOOST_CHECK_EQUAL(slevel2, NOTICE);
}
