/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLMacroProcessor.C
 *    Impl XMLMacroProcessor.h
 *
 * @author Adrian Kriegel
 * @date   Fri Jan 17 2025
 */

#include <xml/macros/XMLMacroProcessor.h>

#include <iterator>

#include <sstream>

#include <serialization/Serialization.h>
#include <serialization/XMLSerializer.h>

#include <utils/MakeString.h>

#include <xml/XMLDomPreprocessor.h>
#include <xml/XMLDomReflect.h>

#include <xml/macros/Types.h>
#include <xml/macros/Utils.h>
#include <xml/macros/Builtins.h>

using namespace mira;
using namespace mira::xmlmacros;

const XMLDom::NameSpace XMLMacroProcessor::NAMESPACE =
    XMLDom::NameSpace("macro", "http://www.mira-project.org/xmlns:macro");

inline bool isPathToXMLFile(const std::string& path)
{
	return path.size() > 4 && path.substr(path.size() - 4) == ".xml";
}

/**
 * Selects the children of the node that are specified by the selector.
 *
 * @param node The node to select parts from.
 * @param path Path selector (dot separated path of node names).
 */
inline void selectParts(XMLNode begin, XMLNode end, const std::string& path)
{
	if (path.empty() || begin == end) {
		return;
	}

	const auto dotPos = path.find('.');

	std::string thisPathElement = dotPos == std::string::npos ? std::move(path) : path.substr(0, dotPos);

	const auto remainingPath = dotPos == std::string::npos ? std::string() : path.substr(dotPos + 1);
	
	XMLDom::iterator targetNode = end;

	for (auto it = begin; it != end; ) {
		if (*it == thisPathElement) {
			targetNode = it;
			++it;
		} else {
			it = it.remove();
		}
	}

	if (targetNode == end) {
		MIRA_THROW(XMacro, "No node matching \"" + thisPathElement + "\"")
	}

	try {
		selectParts(targetNode.begin(), targetNode.end(), remainingPath);
		
		spreadNode(targetNode, targetNode);
	}
	catch (XMacro& e) {
		MIRA_RETHROW(e, "while selecting path \"" + thisPathElement + "\" in " + *targetNode)
	}
}

XMLDom::iterator XMLMacroProcessor::expand(XMLDom::iterator node)
{
	const auto [identifier, selector] = getIdentifier(node);

	MacroPtr handler;

	// If the node is a definition, transform the node so that it is picked up by the Define macro.
	if (DefineMacro::isDefinition(identifier.name)) {
		handler = mDefineHandler;
	}
	// If the macro is imported in-line
	else if (node.has_attribute("From")) {
		handler = mInlineImportHandler;
	}
	// The node is an invocation of a (hopefully) defined macro.
	else {
		handler = getCurrentScope().findOptional(identifier);

		if (handler == nullptr) {
			MIRA_THROW(XInvalidConfig, "No definition for macro \"" + identifier.toString() + "\" found.")
		}
	}

	const auto source = identifier.toString() + " at " + node.uri() + ":" + std::to_string(node.line());

	if (mPreprocessor->expandMacrosOnly()) {
		// insert a comment to indicate expansion of the macro
		node.insert_comment_before("expanded macro " + source);
		node.insert_comment_after("end of expansion of macro " + source);
	}

	// IMacro::expand(...) returns the next node to process, not the expansion result
	// therefore we need to remember the start and end position of node in order to find the inserted nodes afterwards.
	auto parent = node.parent();
	
	const auto distFromBegin = std::distance(node.parent().begin(), node);
	const auto distFromEnd = std::distance(std::next(node), node.parent().end());

	pushInvocation({node.uri(), node.line()});

	XMLDom::iterator nextNode;

	try {
		nextNode = handler->expand(*this, node);
	}
	catch (Exception& e) {
		MakeString info;

		info << "expanding <" << identifier.toString() << "> at " << node.uri() << "(" << node.line() << ")";

		if (auto source = handler->getSource()) {
			info << " defined at " << source->file << "(" << source->line << ")";
		}

		MIRA_RETHROW(e, std::string(info));
	}
	catch (...) {
		throw;
	}

	if (!selector.empty()) {
		auto begin = std::next(parent.begin(), distFromBegin);
		
		XMLNode end = begin;
		// cannot use std::prev() on the end iterator ...
		for ( ; std::distance(end, parent.end()) != distFromEnd; ) { ++end; }

		selectParts(begin, end, selector);
	}

	popInvocation();

	return nextNode;
}

void XMLMacroProcessor::onDocumentEntered(XMLDom::const_iterator root, XMLDom::iterator ioNode)
{
	// Create a fresh scope for the included file.
	auto newScope = std::make_shared<xmlmacros::Scope>(xmlmacros::Scope{mGlobalScope, {}});

	// Swap and store the current scope.
	auto prevScope = swapScope(newScope);
	mImportStack.push(std::move(prevScope));

	// Nothing more to do if no content is included from the document.
	if (ioNode.begin() == ioNode.end()) {
		MIRA_LOG(WARNING) << "Not including macro namespaces as no XML has been included.";
		return;
	}

	// Handle imports from macro prefixes.
	for (auto ns = root.ns_begin(); ns != root.ns_end(); ++ns) {
		const auto prefix = ns.prefix();
		const auto href = ns.href();

		// Only transform namespaces that point to .xml files. Leave everything else untouched.
		if (prefix == "macro" || !isPathToXMLFile(href)) {
			continue;
		}

		// Create an include node for each namespace and let the XMLDomPreprocessor handle actually including
		// the files.
		auto include = ioNode.begin().insert_before("include");
		include.add_attribute("macroNamespace", prefix);
		include.add_attribute("file", href);
		include.setUri(root.uri());
	}
}

void XMLMacroProcessor::onDocumentLeft(const std::string& targetNamespace)
{
	auto prevScope = mImportStack.top();

	if (!targetNamespace.empty()) {
		for (const auto& [name, value] : mCurrentScope->namespaces[XMLMacroProcessor::NAMESPACE.prefix]) {
			prevScope->define({targetNamespace, name}, value);
		}
	}

	swapScope(prevScope);

	mImportStack.pop();
}

bool XMLMacroProcessor::isMacro(const XMLDom::const_iterator& node)
{
	const auto ns = node.nameSpace();

	// Either the namespace is the `macro` namespace or the namespace points to an xml file.
	return ns.href == NAMESPACE.href || isPathToXMLFile(ns.href);
}

XMLNode XMLMacroProcessor::processInPlace(XMLNode node)
{
	// preprocessAll will actually only process the children, not the node itself ...
	// so we have to create a dummy node, attach the node as a child and remove the original node
	XMLDom dummy;
	auto newNode = dummy.root().add_child("dummy");
	newNode.replace(node);

	XMLDom::iterator root = dummy.root();

	mPreprocessor->preprocessAll(root);

	// root had exactly one child before but may have more or less now
	for (auto it = root.begin(); it != root.end(); ++it) {
		node.insert_after(it);
	}

	return node.remove();
}

std::string XMLMacroProcessor::coerceToString(const std::string& identifier)
{
	auto colPos = identifier.find(':');

	const std::string xmlns = colPos == std::string::npos ? NAMESPACE.prefix : identifier.substr(0, colPos);
	const std::string name = colPos == std::string::npos ? identifier : identifier.substr(colPos + 1);

	auto handler = getCurrentScope().find({xmlns, name});

	if (!handler->canCoerceToString()) {
		Identifier id{xmlns, name};

		MIRA_THROW(XMacro,
		           "Macro \"" << id.toString()
		                      << "\" requires parameters and cannot directly be coerced to string.")
	}

	std::stringstream result;

	handler->coerceToString(*this, result);
	return result.str();
}

void XMLMacroProcessor::pushInvocation(xmlmacros::SourceInfo&& info)
{
	if (mInvokeStack.size() >= MAX_INVOKE_DEPTH) {
		MIRA_THROW(XMaxDepth, "Reached max. macro invocation depth.");
	}

	mInvokeStack.emplace_back(std::move(info));
}

void XMLMacroProcessor::popInvocation()
{
	mInvokeStack.pop_back();
}

XMLMacroProcessor::XMLMacroProcessor(XMLDomPreprocessor& preprocessor)
: mGlobalScope(std::make_shared<Scope>(Scope{nullptr, {}})),
  // Attach a new scope which represents the entry point file.
  mCurrentScope(std::make_shared<Scope>(Scope{mGlobalScope, {}})), //
  mPreprocessor(&preprocessor),                                    //
  mDefineHandler(std::make_shared<DefineMacro>()),                 //
  mInlineImportHandler(std::make_shared<ImportInlineMacro>())
{
	mGlobalScope->namespaces[NAMESPACE.prefix] = {
	    {ForMacro::NAME, std::make_shared<ForMacro>()},           //
		{ZipMacro::NAME, std::make_shared<ZipMacro>()},           //
		{SplitMacro::NAME, std::make_shared<SplitMacro>()},       //
	    {PrintXMLMacro::NAME, std::make_shared<PrintXMLMacro>()}, //
	    {NothingMacro::NAME, std::make_shared<NothingMacro>()},   //
	};

	mPreprocessor->registerXMLVariableCallback(
	    "macro", [this](const std::string& name) { return coerceToString(name); });
}
