/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ResolveVariable.C
 *    Implementation of ResolveVariable.h.
 *
 * @author Tim Langner
 * @date   2011/11/18
 */

#include <utils/ResolveVariable.h>

#include <boost/regex.hpp>

#include <utils/PathFinder.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

std::string resolveVariables(std::string s, 
                             boost::function<std::string (const std::string& var)> functor,
                             bool ignoreErrors)
{
	// replace all occurrences of variables by their content
	std::string::iterator start, end;
	std::vector<boost::regex> regexps(3);
	regexps[0] = boost::regex("\\x24(\\w+)");			// matches $ENV_NAME
	regexps[1] = boost::regex("\\x24\\x7B(\\w+)\\x7D");	// matches ${ENV_NAME}
	regexps[2] = boost::regex("\\x25(\\w+)\\x25");		// matches %ENV_NAME%
	boost::match_results<std::string::iterator> results;
	for ( uint32 i=0; i<regexps.size(); ++i)
	{
		start = s.begin();
		end = s.end();
		boost::match_flag_type flags = boost::match_default;
		while(regex_search(start, end, results, regexps[i], flags))
		{
			std::string varname(results[1].first, results[1].second);
			std::string value;
			try
			{
				value = functor(varname);
				s.replace(results[0].first, results[0].second, value);
				start = s.begin();
				end = s.end();
			}
			catch(Exception&)
			{
				if (ignoreErrors)
					start = results[0].second;
				else
					throw;
			}
			flags |= boost::match_prev_avail | boost::match_not_bob;
		}
	}
	return s;
}

///////////////////////////////////////////////////////////////////////////////

std::string resolveSpecialVariables(std::string s,
                                    boost::function<std::string (const std::string&,
                                                                 const std::string&)> functor,
                                    bool ignoreErrors)
{
	// replace all occurrences of find
	std::string::iterator start, end;
	boost::regex regexps = boost::regex(
		"\\x24"						// $
		"\\x7B"						// {
		"([^\\x24\\x7B\\x7D\\s]*)"	// PATTERN all chars except {}$ and space
		"\\s"						// space
		"([^\\x24\\x7B\\x7D]*)"		// VARIABLE all chars except {} and $
		"\\x7D");					// }
	boost::match_results<std::string::iterator> results;

	start = s.begin();
	end = s.end();
	boost::match_flag_type flags = boost::match_default;
	while(regex_search(start, end, results, regexps, flags))
	{
		std::string pattern(std::string(results[1].first, results[1].second));
		std::string varname(std::string(results[2].first, results[2].second));
		std::string value;
		try
		{
			value = functor(pattern, varname);
			s.replace(results[0].first, results[0].second, value);
			start = s.begin();
			end = s.end();
		}
		catch(Exception&)
		{
			if (ignoreErrors)
				start = results[0].second;
			else
				throw;
		}
		flags |= boost::match_prev_avail | boost::match_not_bob;
	}
	return s;
}

//////////////////////////////////////////////////////////////////////////////

std::string resolveSpecialVariables(const std::string& pattern, std::string s,
                                    boost::function<std::string (const std::string& var)> functor,
                                    bool ignoreErrors)
{
	std::string::iterator start, end;
	std::string rgs =
		"\\x24"					// $
		"\\x7B";				// {
	rgs += pattern;				// PATTERN
	rgs += "\\s"				// space
		"([^\\x24\\x7B\\x7D]*)"	// VARIABLE all chars except {} and $
		"\\x7D";				// }
	boost::regex regexps = boost::regex(rgs.c_str());	// matches ${pattern variable}
	boost::match_results<std::string::iterator> results;

	start = s.begin();
	end = s.end();
	boost::match_flag_type flags = boost::match_default;
	while(regex_search(start, end, results, regexps, flags))
	{
		std::string varname(std::string(results[1].first, results[1].second));
		std::string value;
		try
		{
			value = functor(varname);
			s.replace(results[0].first, results[0].second, value);
			start = s.begin();
			end = s.end();
		}
		catch(Exception&)
		{
			if (ignoreErrors)
				start = results[0].second;
			else
				throw;
		}
		flags |= boost::match_prev_avail | boost::match_not_bob;
	}
	return s;
}

//////////////////////////////////////////////////////////////////////////////

}
