/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ProcessAncestryHelper.C
 *    Implementation of ProcessAncestryHelper.h
 *
 * @author Christof Schroeter, Christian Martin
 * @date   2024/10/28
 */

#include "ProcessAncestryHelper.h"

#include <error/Exceptions.h>
#include <utils/Time.h>

#if MIRA_USE_LIBPROC2
# include <libproc2/pids.h>
#else
# include <proc/readproc.h>
#endif

namespace mira { namespace priv {

//////////////////////////////////////////////////////////////////////////////

std::vector<int> getChildProcesses(int pid, int depth, int maxDepth,
                                   const std::map<int, std::vector<int>>& ancestry)
{
	std::vector<int> res;

	if (!ancestry.count(pid))
		return res;

	foreach (int p, ancestry.at(pid)) {
		res.push_back(p);
		if ((maxDepth < 0) || (depth < maxDepth)) {
			std::vector<int> resNextLevel = getChildProcesses(p, depth+1, maxDepth, ancestry);
			res.insert(res.end(), resNextLevel.begin(), resNextLevel.end());
		}
	}

	return res;
}

//////////////////////////////////////////////////////////////////////////////

#ifdef MIRA_USE_LIBPROC2

// use libproc2
std::map<int, std::vector<int>> getProcessAncestry()
{
	std::map<int, std::vector<int>> ancestry;

	struct pids_info* pinfo = NULL;
	enum pids_item pinfoItems[] = { PIDS_ID_PID, PIDS_ID_PPID };
	if (procps_pids_new(&pinfo, pinfoItems, 2) < 0)
		MIRA_THROW(XSystemCall, "Failed to call procps_pids_new");

	struct pids_stack* ps;
	while ((ps = procps_pids_get(pinfo, PIDS_FETCH_TASKS_ONLY))) {
		int pid  = PIDS_VAL(0, s_int, ps, pinfo);
		int ppid = PIDS_VAL(1, s_int, ps, pinfo);

		ancestry[ppid].push_back(pid);
		ancestry[pid]; // insert the child (with empty child list), if it does not>
	}

	procps_pids_unref(&pinfo);
	return ancestry;
}

#else // MIRA_USE_LIBPROC2

// use libprocps
std::map<int, std::vector<int>> getProcessAncestry()
{
	PROCTAB* tab = NULL;
	int count = 0;
	while (!tab) {
		tab = ::openproc(PROC_FILLSTATUS | PROC_FILLSTAT);
		MIRA_SLEEP(10);
		++count;
	}

	std::map<int, std::vector<int>> ancestry;

	for (;;) {
		proc_t* proc = ::readproc(tab, NULL);
		if (!proc)
			break;

		ancestry[proc->ppid].push_back(proc->tid);
		ancestry[proc->tid]; // insert the child (with empty child list), if it does not exist yet

		freeproc(proc);
	}

	closeproc(tab);
	return ancestry;
}

#endif // MIRA_USE_LIBPROC2

//////////////////////////////////////////////////////////////////////////////

}}
