/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file xml/macros/Builtins.h
 *    Default Macros.
 *
 * @author Adrian Kriegel
 * @date   Fri Jan 17 2025
 */

#ifndef _MIRA_XML_MACROS_BUILTINS_H_
#define _MIRA_XML_MACROS_BUILTINS_H_

#include <xml/macros/Utils.h>
#include <xml/macros/Types.h>
#include <xml/macros/IMacro.h>

#include <serialization/adapters/std/vector>

namespace mira::xmlmacros {

/**
 * The "Define" macro.
 * Allows to define new macros inside of XML files. @see UserDefinedMacro
 *
 * Usage:
 *
 * <macro:Define_MyMacro>
 *   <Parameters>
 *     <MyParameter />
 *     <MyParameterWithDefault>42</MyParameterWithDefault>
 *   </Parameters>
 *   <Body>
 *     <!-- Macro body. -->
 *   </Body>
 * </macro:Define_MyMacro>
 */
class DefineMacro : public IMacro
{
public:
	static constexpr const char* PREFIX = "Define_";

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;

	static bool isDefinition(const std::string& name)
	{
		return name.rfind(PREFIX, 0) == 0;
	}

	static std::string extractName(XMLDom::const_iterator node);
}; // class DefineMacro

/**
 * Usage: <macro:MyImportedMacro From="file/that/defines/MyImportedMacro.xml"> ...
 */
class ImportInlineMacro : public IMacro
{
public:
	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;

protected:
	uint mNameCount;

	std::string createAnonymousName(const std::string originalName)
	{
		return "ImportedMacro_" + originalName + "_" + std::to_string(mNameCount++);
	}
}; // class ImportMacro

class ForMacro : public IMacro
{
public:
	static constexpr const char* NAME = "For";

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;

}; // class ForMacro

/**
 * Splits content into words and inserts them as <item>word</item> nodes.
 */
class SplitMacro : public IMacro
{
public:

	static constexpr const char* NAME = "Split";

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;

}; // class SplitMacro

/**
 * Zips lists. Example:
 * <macro:Zip>
 *     <Foo>
 *         <item>1</item>
 *         <item>2</item>
 *         <item>3</item>
 *     </Foo>
 *     <Bar>
 *         <item>A</item>
 *         <item>B</item>
 *         <item>C</item>
 *     </Bar>
 * </macro:Zip>
 * 
 * will result in:
 * 
 * <item>
 *     <Foo>1</Foo>
 *     <Bar>A</Bar>
 * </item>
 * <item>
 *     <Foo>2</Foo>
 *     <Bar>B</Bar>
 * </item>
 * <item>
 *     <Foo>3</Foo>
 *     <Bar>C</Bar>
 * </item>
 */
class ZipMacro : public IMacro
{
public:
	struct ZipBody {

		struct ZipElement {
			std::string name;
			std::vector<CopyableXMLDom> items;

			void reflect(XMLDeserializer& r) {
				name = *r.getNode();
				r.delegate(items);
			}
		};

		std::vector<ZipElement> elements;

		void reflect(XMLDeserializer& r) {

			elements.clear();

			auto node = r.getNode();

			elements.reserve(std::distance(node.begin(), node.end()));

			for (auto it = node.begin(); it != node.end(); ++it) {
				r.property((*it).c_str(), elements.emplace_back(), "");
			}
		}
	}; // struct ZipBody

	static constexpr const char* NAME = "Zip";

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;
}; // class ZipMacro

class NothingMacro : public IMacro
{
public:
	static constexpr const char* NAME = "Nothing";

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final
	{
		return node.remove();
	}

	[[nodiscard]] bool canCoerceToString() const final
	{
		return true;
	}

	void coerceToString(XMLMacroProcessor& processor, std::ostream& ss) final
	{
		ss << "";
	}
}; // class NothingMacro

/**
 * Evaluates children and prints the result to stdout for debugging.
 */
class PrintXMLMacro : public IMacro
{
public:
	static constexpr const char* NAME = "PrintXML";

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;
}; // class PrintXMLMacro

}; // namespace mira::xmlmacros

#endif // _MIRA_XML_MACROS_BUILTINS_H_
