/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TimeHumanReadable.h
 *    Extension of time types for human-readable serialization
 *
 * @author Christof Schröter
 * @date   2024/09/19
 */

#ifndef _MIRA_TIMEHUMANREADABLE_H_
#define _MIRA_TIMEHUMANREADABLE_H_

#include <utils/Time.h>
#include <error/Exceptions.h>

#include <serialization/ReflectControlFlags.h>
#include <serialization/Accessor.h>
#include <serialization/PropertySerializer.h>

#include <utils/IsCheapToCopy.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 *  The serialization of time-related types Time, Date and Duration is primarily
 *  aimed at efficiency and precision. They may be less suited for use cases
 *  where the serialized output should be read by humans, or input for
 *  deserialization should be edited by humans.
 *
 *  For such cases, we add here derivations of mira::Time, mira::Date and
 *  mira::Duration which change the serialization format. These can be used
 *  for variables where the form of the output format matters more than
 *  precision or performance.
 *
 *  For each type, there is one variant serializing to a compact string
 *  (based on ISO 8601), and one that serializes to separate primitive members
 *  like 'Year', 'Month', 'Hours' etc.
 *
 *  Instead of using objects of these classes as replacement for
 *  Time, Date or Duration, specific getters and setters can be applied to
 *  objects of the original types just for serialization (defined within
 *  the owning class' reflect() method). Such setters and getters are also
 *  provided here, they use the alternative classes as proxies for serialization.
 */
namespace HumanReadableSerialization {

/**
 * Derivation of mira::Date with human-readable serialization
 *
 * Specialized as \ref DateSerializedAsIsoString, \ref DateSerializedAsYMD
 */
template<bool AsString = true>
class Date : public mira::Date {
public:
	typedef mira::Date Base;
public:
#if !defined(DATE_TIME_NO_DEFAULT_CONSTRUCTOR)
	Date() {}
#endif
	Date(const Base& date) : Base(date) {}
	using mira::Date::Date; // inherit all constructors
};

} // namespace HumanReadableSerialization

///////////////////////////////////////////////////////////////////////////////

/// Derivation of mira::Date with serialization as string
using DateSerializedAsIsoString = HumanReadableSerialization::Date<true>;

/// Derivation of mira::Date with serialization as year/month/day members
using DateSerializedAsYMD = HumanReadableSerialization::Date<false>;

///////////////////////////////////////////////////////////////////////////////

template <>
class IsCheapToCopy<DateSerializedAsIsoString> : public std::true_type {};

template <>
class IsCheapToCopy<DateSerializedAsYMD> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

MIRA_SPLIT_REFLECT(DateSerializedAsIsoString)
MIRA_SPLIT_REFLECT(DateSerializedAsYMD)

template<typename Reflector>
void reflectRead(Reflector& r, DateSerializedAsIsoString& date)
{
	std::string s = to_iso_extended_string(date);
	r.delegate(s);
}

template<typename Reflector>
void reflectRead(Reflector& r, DateSerializedAsYMD& date)
{
	DateSerializedAsYMD::ymd_type ymd = date.year_month_day();
	int y = ymd.year;
	int m = ymd.month;
	int d = ymd.day;
	r.member("Year",  y, "year");
	r.member("Month", m, "month");
	r.member("Day",   d, "day");

	r.roproperty("Weekday", boost::lexical_cast<std::string>(date.day_of_week()),
	             "Day of week");
}

template<typename Reflector>
void reflectWrite(Reflector& r, DateSerializedAsIsoString& date)
{
	std::string s;
	r.delegate(s);
	date = boost::gregorian::from_string(s);
}

template<typename Reflector>
void reflectWrite(Reflector& r, DateSerializedAsYMD& date)
{
	int y, m, d;
	r.member("Year",  y, "");
	r.member("Month", m, "");
	r.member("Day",   d, "");

	date = DateSerializedAsYMD::Base(y, m, d);
}

void reflect(PropertySerializer& r, DateSerializedAsYMD& date);

///////////////////////////////////////////////////////////////////////////////

namespace HumanReadableSerialization {

/**
 * Derivation of mira::Duration with human-readable serialization
 *
 * Specialized as \ref DurationSerializedAsIsoString, \ref DurationSerializedAsHMS
 */
template<bool AsString = true>
class Duration : public mira::Duration {
public:
	typedef mira::Duration Base;
#if !defined(DATE_TIME_NO_DEFAULT_CONSTRUCTOR)
	Duration() {}
#endif
	Duration(const Base& date) : Base(date) {}
	using mira::Duration::Duration; // inherit all constructors

	MIRA_SPLIT_REFLECT_MEMBER

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		if constexpr (AsString) {
			std::string s = to_simple_string(*this);
			r.delegate(s);
		}
		else {
			int h = hours();
			int m = minutes();
			int s = seconds();
			int us = milliseconds() * 1000ll + microseconds();
			r.member("Hours",        h,  "Hours 0..23");
			r.member("Minutes",      m,  "Minutes 0..59");
			r.member("Seconds",      s,  "Seconds 0..59");
			r.member("MicroSeconds", us, "Microseconds 0..999999 (optional, default=0)");
		}
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		if constexpr (AsString) {
			std::string s;
			r.delegate(s);
			if (s.find(':') == std::string::npos) {
				MIRA_THROW(XIO, "DurationSerializedAsIsoString: Duration string for deserialization "
				                "expects format 'hh:mm[:ss[.frac]]'");
			}
			*this = Base(boost::posix_time::duration_from_string(s));
		} else {
			int64 h, m, s, us;
			r.member("Hours",        h, "Hours 0..23");
			r.member("Minutes",      m, "Minutes 0..59");
			r.member("Seconds",      s, "Seconds 0..59");
			r.member("MicroSeconds", us, "Microseconds 0..999999 (optional, default=0)", 0);

			*this = Base(h, m, s, us);
		}
	}

	void reflect(PropertySerializer& r)
	{
		if constexpr (AsString) {
			reflectRead(r);
		}
		else {
			r.property("Positive",
			           getter<bool>([this]{ return !is_negative(); }),
			           setter<bool>([this](bool p){ if (p == is_negative()) *this = invert_sign(); }),
			           "Is positive?");
			r.property("Hours",
			           getter<int>([this]{ return std::abs(hours()); }),
			           setter<int>([this](int h){
			             int hs = (is_negative() ? -h : h);
			             *this = Base(hs, minutes(), seconds(),
			                          milliseconds()*1000ll + microseconds());
			           }),
			           "Hours", PropertyHints::minimum(0));
			r.property("Minutes",
			           getter<int>([this]{ return std::abs(minutes()); }),
			           setter<int>([this](int m){
			             int ms = (is_negative() ? -m : m);
			             *this = Base(hours(), ms, seconds(),
			                          milliseconds()*1000ll + microseconds());
			           }),
			           "Minutes", PropertyHints::limits(0, 59));
			r.property("Seconds",
			           getter<int>([this]{ return std::abs(seconds()); }),
			           setter<int>([this](int s){
			             int ss = (is_negative() ? -s : s);
			             *this = Base(hours(), minutes(), ss,
			                          milliseconds()*1000ll + microseconds());
			           }),
			           "Seconds", PropertyHints::limits(0, 59));
			r.property("Microseconds",
			           getter<int>([this]{ return std::abs(milliseconds()) * 1000ll
			                                    + std::abs(microseconds()); }),
			           setter<int>([this](int us){
			             int uss = (is_negative() ? -us : us);
			             *this = Base(hours(), minutes(), seconds(), uss);
			           }),
			           "Microseconds", PropertyHints::limits(0ll, 999999ll));
		}
	}
};

} // namespace HumanReadableSerialization

///////////////////////////////////////////////////////////////////////////////

/// Derivation of mira::Duration with serialization as string
using DurationSerializedAsIsoString = HumanReadableSerialization::Duration<true>;

/// Derivation of mira::Duration with serialization as hours/minutes/seconds/milli-/microseconds members
using DurationSerializedAsHMS = HumanReadableSerialization::Duration<false>;

///////////////////////////////////////////////////////////////////////////////

template <>
class IsCheapToCopy<DurationSerializedAsIsoString> : public std::true_type {};

template <>
class IsCheapToCopy<DurationSerializedAsHMS> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

namespace HumanReadableSerialization {

/**
 * Derivation of mira::Time with human-readable serialization
 *
 * Specialized as \ref TimeSerializedAsIsoString, \ref TimeSerializedAsYMDHMS
 */
template<bool AsString = true>
class Time : public mira::Time {
public:
	typedef mira::Time Base;
#if !defined(DATE_TIME_NO_DEFAULT_CONSTRUCTOR)
	Time() {}
#endif
	Time(const Base& date) : Base(date) {}
	using mira::Time::Time; // inherit all constructors

	MIRA_SPLIT_REFLECT_MEMBER

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		if constexpr (AsString) {
			std::string s = to_iso_extended_string(*this);
			r.delegate(s);
		} else {
			DateSerializedAsYMD dateHR(date());
			mira::reflectRead(r, dateHR);
			DurationSerializedAsHMS(time_of_day()).reflectRead(r);
		}
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		if constexpr (AsString) {
			std::string s;
			r.delegate(s);
			if (s.find('T') == std::string::npos) {
				MIRA_THROW(XIO, "TimeSerializedAsIsoString: Time string for deserialization "
				                "expects format 'yyyy-mm-ddThh:mm:ss[.frac]'");
			}
#if BOOST_VERSION >= 107300
			*this = Base(boost::posix_time::from_iso_extended_string(s));
#else
			*this = Base(boost::date_time::parse_delimited_time<Base>(s, 'T'));
#endif
		} else {
			DateSerializedAsYMD date;
			mira::reflectWrite(r, date);

			DurationSerializedAsHMS timeOfDay;
			timeOfDay.reflectWrite(r);

			*this = Base(date, timeOfDay);
		}
	}

	void reflect(PropertySerializer& r)
	{
		if constexpr (AsString) {
			reflectRead(r);
		}
		else {
			using mira::Date;
			using mira::Duration;

			r.property("Year",
			           getter<int>([this]{ return date().year();}),
			           setter<int>([this](int y){
			             auto d = date();
			             *this = Base(Date(y, d.month(), d.day()),
			                          time_of_day());
			           }),
			           "Year");
			r.property("Month",
			           getter<int>([this]{ return date().month();}),
			           setter<int>([this](int m){
			             auto d = date();
			             *this = Base(Date(d.year(), m, d.day()),
			                          time_of_day());
			           }),
			           "Month", PropertyHints::limits(1, 12));
			r.property("Day",
			           getter<int>([this]{ return date().day();}),
			           setter<int>([this](int dy){
			             auto d = date();
			             *this = Base(Date(d.year(), d.month(), dy),
			                          time_of_day());
			           }),
			           "Day", PropertyHints::limits(1, 31));
			r.roproperty("Weekday",
			             getter<std::string>([this]{
			               return boost::lexical_cast<std::string>(date().day_of_week());
			             }),
			             "Day of week");
			r.property("Hours",
			           getter<int>([this]{ return time_of_day().hours(); }),
			           setter<int>([this](int h){
			             auto t = time_of_day();
			             *this = Base(date(), Duration(h, t.minutes(), t.seconds(),
			                                           t.total_microseconds() % 1000000ll));
			           }),
			           "Hours", PropertyHints::limits(0, 23));
			r.property("Minutes",
			           getter<int>([this]{ return time_of_day().minutes(); }),
			           setter<int>([this](int m){
			             auto t = time_of_day();
			             *this = Base(date(), Duration(t.hours(), m, t.seconds(),
			                                           t.total_microseconds() % 1000000ll));
			           }),
			           "Minutes", PropertyHints::limits(0, 59));
			r.property("Seconds",
			           getter<int>([this]{ return time_of_day().seconds(); }),
			           setter<int>([this](int s){
			             auto t = time_of_day();
			             *this = Base(date(), Duration(t.hours(), t.minutes(), s,
			                                           t.total_microseconds() % 1000000ll));
			           }),
			           "Seconds", PropertyHints::limits(0, 59));
			r.property("Microseconds",
			           getter<int>([this]{ return time_of_day().total_microseconds() % 1000000ll; }),
			           setter<int>([this](int us){
			             auto t = time_of_day();
			             *this = Base(date(), Duration(t.hours(), t.minutes(), t.seconds(), us));
			           }),
			           "Microseconds", PropertyHints::limits(0ll, 999999ll));
		}
	}
};

} // namespace HumanReadableSerialization

///////////////////////////////////////////////////////////////////////////////

/**
 * Derivation of mira::Time with serialization as string.
 *
 * @note: Time (based on boost::posix_time::ptime) is timezone-agnostic, i.e. the represented time
 * is a time point in a continuous time frame, but it has no knowledge/makes no assumption
 * about the reference time zone. Therefore, serialization does not produce and
 * deserialization does not recognize/interpret any time zone information.
 */
using TimeSerializedAsIsoString = HumanReadableSerialization::Time<true>;

/// Derivation of mira::Time with serialization as year/month/day/hour/... members
using TimeSerializedAsYMDHMS = HumanReadableSerialization::Time<false>;

///////////////////////////////////////////////////////////////////////////////

template<typename SerializerTag>
class IsTransparentSerializable<DateSerializedAsIsoString, SerializerTag> : public std::true_type {};

template<typename SerializerTag>
class IsTransparentSerializable<DurationSerializedAsIsoString, SerializerTag> : public std::true_type {};

template<typename SerializerTag>
class IsTransparentSerializable<TimeSerializedAsIsoString ,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

template <>
class IsCheapToCopy<TimeSerializedAsIsoString> : public std::true_type {};

template <>
class IsCheapToCopy<TimeSerializedAsYMDHMS> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

/**
 * A tag type used as parameter type in humanReadableGetter, humanReadableSetter,
 * humanReadableAccessor to select serialization as individual components (year/month/day etc.)
 */
struct TimeTypesSerializedAsElements { explicit TimeTypesSerializedAsElements() = default; };

/**
 * In addition to classes which can replace Date/Duration/Time,
 * there are getters/setters/accessors for objects of these plain types.
 * Usage example:
 * \code
 * struct MyClass
 * {
 *   template<typename Reflector>
 *   void reflect(Reflector& r) {
 *     auto acc1 = humanReadableAccessor(time1);
 *     r.member("TimeString", acc1, "");
 *     auto acc2 = humanReadableAccessor(time2, TimeTypesSerializedAsElements());
 *     r.member("TimeVerbose", acc2, "");
 *   }
 *   Time time1;
 *   Time time2;
 * };
 * \endcode
 */

/// Getter for Date using DateSerializedAsIsoString proxy
Getter<DateSerializedAsIsoString> humanReadableGetter(const Date& date);

/// Setter for Date using DateSerializedAsIsoString proxy
Setter<DateSerializedAsIsoString> humanReadableSetter(Date& date);

/// Accessor for Date using DateSerializedAsIsoString proxy
Accessor<Getter<DateSerializedAsIsoString>, Setter<DateSerializedAsIsoString>>
humanReadableAccessor(Date& date);

/// Getter for Date using DateSerializedAsYMD proxy
Getter<DateSerializedAsYMD> humanReadableGetter(const Date& date, TimeTypesSerializedAsElements);

/// Setter for Date using DateSerializedAsYMD proxy
Setter<DateSerializedAsYMD> humanReadableSetter(Date& date, TimeTypesSerializedAsElements);

/// Accessor for Date using DateSerializedAsYMD proxy
Accessor<Getter<DateSerializedAsYMD>, Setter<DateSerializedAsYMD>>
humanReadableAccessor(Date& date, TimeTypesSerializedAsElements);

/// Getter for Duration using DurationSerializedAsIsoString proxy
Getter<DurationSerializedAsIsoString> humanReadableGetter(const Duration& duration);

/// Setter for Duration using DurationSerializedAsIsoString proxy
Setter<DurationSerializedAsIsoString> humanReadableSetter(Duration& duration);

/// Accessor for Duration using DurationSerializedAsIsoString proxy
Accessor<Getter<DurationSerializedAsIsoString>, Setter<DurationSerializedAsIsoString>>
humanReadableAccessor(Duration& duration);

/// Getter for Duration using DurationSerializedAsHMS proxy
Getter<DurationSerializedAsHMS> humanReadableGetter(const Duration& duration, TimeTypesSerializedAsElements);

/// Setter for Duration using DurationSerializedAsHMS proxy
Setter<DurationSerializedAsHMS> humanReadableSetter(Duration& duration, TimeTypesSerializedAsElements);

/// Accessor for Duration using DurationSerializedAsHMS proxy
Accessor<Getter<DurationSerializedAsHMS>, Setter<DurationSerializedAsHMS>>
humanReadableAccessor(Duration& duration, TimeTypesSerializedAsElements);

/// Getter for Time using TimeSerializedAsIsoString proxy
Getter<TimeSerializedAsIsoString> humanReadableGetter(const Time& time);

/// Setter for Time using TimeSerializedAsIsoString proxy
Setter<TimeSerializedAsIsoString> humanReadableSetter(Time& time);

/// Accessor for Time using TimeSerializedAsIsoString proxy
Accessor<Getter<TimeSerializedAsIsoString>, Setter<TimeSerializedAsIsoString>>
humanReadableAccessor(Time& time);

/// Getter for Time using TimeSerializedAsYMDHMS proxy
Getter<TimeSerializedAsYMDHMS> humanReadableGetter(const Time& time, TimeTypesSerializedAsElements);

/// Setter for Time using TimeSerializedAsYMDHMS proxy
Setter<TimeSerializedAsYMDHMS> humanReadableSetter(Time& time, TimeTypesSerializedAsElements);

/// Accessor for Time using TimeSerializedAsYMDHMS proxy
Accessor<Getter<TimeSerializedAsYMDHMS>, Setter<TimeSerializedAsYMDHMS>>
humanReadableAccessor(Time& time, TimeTypesSerializedAsElements);

///////////////////////////////////////////////////////////////////////////////

}

#endif
