/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SharedLibrary.h
 *    Classes for loading shared libraries.
 *
 * @author Tim Langner
 * @date   2010/10/01
 */

#ifndef _MIRA_SHAREDLIBRARY_H_
#define _MIRA_SHAREDLIBRARY_H_

#include <list>

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#endif

#include <utils/Path.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Adds the system dependent library pre- and suffix to the given name if
 * necessary.
 * @param[in] path The path to be resolved
 * @return The path with system dependent library pre and suffix
 */
Path resolveLibraryName(const Path& path);

///////////////////////////////////////////////////////////////////////////////

/**
 * Class for loading a single shared library.
 * The library is loaded/opened on construction and closed on destruction.
 */
class SharedLibrary : boost::noncopyable
{
public:
	typedef void* Handle;
	typedef void* Symbol;

	/**
	 * Loads the specified library. Afterwards its symbols can be accessed
	 * using the getSymbol() method.
	 * @param[in] library The relative or absolute path of the shared library.
	 * @throw XIO If the library can not be found or loaded.
	 */
	SharedLibrary(const Path& library);

	~SharedLibrary();

	/**
	 * Return the name prefix of libraries (e.g. 'lib' for unix shared libraries).
	 * The full library name is:
	 *    <prefix>NAME<postfix><extension>
	 */
	static const char* getLibPrefix();

	/**
	 * Returns the name postfix of libraries
	 * The full library name is:
	 *    <prefix>NAME<postfix><extension>
	 */
	static const char* getLibPostfix();

	/**
	 * Return library extension
	 * The full library name is:
	 *    <prefix>NAME<postfix><extension>
	 */
	static const char* getLibExtension();

	/**
	 * Return true if the shared library is currently open.
	 * @return true if open
	 */
	bool isOpen() const
	{
		return mHandle != 0;
	}

	/**
	 * Close the shared library.
	 */
	void close()
	{
		closeSharedLibrary(mHandle);
		mHandle = 0;
	}

	/**
	 * Get a symbol from the library.
	 * @throw XIO when obtaining the symbol failed.
	 * @param[in] symbol The name of the symbol
	 * @return Handle to the symbol
	 */
	Symbol getSymbol(const std::string& symbol)
	{
		return getSymbol(mHandle, symbol.c_str());
	}

	/// low level opening of libraries without using the class wrapper
	static SharedLibrary::Handle loadSharedLibrary(const char* library);
	
	/// low level getting symbols from libraries without using the class wrapper
	static SharedLibrary::Symbol getSymbol(SharedLibrary::Handle handle, 
	                                       const char* function);

	/// low level closing of libraries without using the class wrapper
	static void closeSharedLibrary(SharedLibrary::Handle handle);

private:
	Handle mHandle;	///< Handle to the opened library
};

///////////////////////////////////////////////////////////////////////////////

typedef boost::shared_ptr<SharedLibrary> SharedLibraryPtr;

/**
 * Class for loading multiple shared libraries.
 * The class keeps track of the opened libraries and closes them upon destruction.
 */
class SharedLibraryLoader
{
public:
	/**
	 * Load a library and open it.
	 * If the specified path does not contain a library extension, the
	 * operating system dependent library prefix and suffix are added to
	 * the library filename. This allows to specify libraries in an
	 * OS independent way.
	 * @param[in] library Path to the library file
	 *
	 */
	void loadLibrary(const Path& library);

	/**
	 * Load all libraries from a directory and open them.
	 * @param[in] path Directory of the library files
	 */
	void loadLibraries(const Path& path);

protected:
	std::list<SharedLibraryPtr> mLibraries;	///< list of all loaded libraries
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
