/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Functionals.h
 *    Collection of functors.
 *
 * @author Tim Langner
 * @date   2010/10/04
 */

#ifndef _MIRA_FUNCTIONALS_H_
#define _MIRA_FUNCTIONALS_H_

#include <functional>
#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private {

template <typename P, typename Op>
struct EvalFirstOf
{
	EvalFirstOf(const Op& iOp, const typename P::first_type& iValue) :
		op(iOp),
		value(iValue)
	{}

	bool operator()(const P& p)
	{
		return op(p.first, value);
	}

	Op op;
	typename P::first_type value;
};

template <typename P, typename Op>
struct EvalSecondOf
{
	EvalSecondOf(const Op& iOp, const typename P::second_type& iValue) :
		op(iOp),
		value(iValue) {}

	bool operator()(const P& p)
	{
		return op(p.second, value);
	}

	Op op;
	typename P::second_type value;
};

}	// namespace
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Functor for using STL functions like find_if, erase_if...
 * with STL collections of pairs.
 * \code
 *     typedef std::pair<int, float> MyPair;
 *     std::vector<MyPair> v;
 *     // find entry in vector where v[x].first == 3
 *     std::find_if(v.begin(), v.end(),
 *                  evalFirstOf<MyPair>(std::equal_to<int>(), 3));
 * \endcode
 * @param[in] iOp The compare functor
 * @param[in] iValue The value to compare to
 * @return An EvalFirstOf functor
 */
template <typename P, typename Op>
Private::EvalFirstOf<P, Op> evalFirstOf(const Op& iOp, 
                                        const typename P::first_type& iValue)
{
	return Private::EvalFirstOf<P,Op>(iOp, iValue);
}

/**
 * Functor for using stl functions like find_if, erase_if...
 * with stl collections of pairs.
 * \code
 *     typedef std::pair<int, float> MyPair;
 *     std::vector<MyPair> v;
 *     // find entry in vector where v[x].second == 1.5f
 *     std::find_if(v.begin(), v.end(),
 *                  evalSecondOf<MyPair>(std::equal_to<float>(), 1.5f));
 * \endcode
 * @param[in] iOp The compare functor
 * @param[in] iValue The value to compare to
 * @return An EvalSecondOf functor
 */
template <typename P, typename Op>
Private::EvalSecondOf<P, Op> evalSecondOf(const Op& iOp, 
                                          const typename P::second_type& iValue)
{
	return Private::EvalSecondOf<P,Op>(iOp, iValue);
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
