/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformerNode.h
 *    AbstractTransformerNode base class and TransformerNode reference
 *    implementation.
 *
 * @author Erik Einhorn
 * @date   2010/10/10
 */

#ifndef _MIRA_TRANSFORMERNODE_H_
#define _MIRA_TRANSFORMERNODE_H_

#include <string>

#ifndef Q_MOC_RUN
#include <boost/iterator/transform_iterator.hpp>
#endif

#include <utils/Stamped.h>
#include <utils/Time.h>

#include <utils/StampedDataQueue.h>
#include <utils/IteratorRangeContainer.h>

#include <transform/RigidTransform.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

// forward decls
class TransformerBase;
class NearestNeighborInterpolator;

/**
 * @ingroup TransformModule
 *
 * Abstract base class where all other different types of transformation
 * nodes must be derived from. Each node can take part in a transformation tree
 * that is managed by the Transformer (GenericTransformer).
 *
 * Moreover, each node has a unique ID within the Transformer it belongs to.
 * Each node stores a pointer to its parent to its children within the
 * transformation tree. Last but not least, each node stores a transformation
 * history. Hence, the transformation of the node at a certain time stamp can
 * be queried using the getTransform() method. Note, that the derived classes
 * need to implement the getTransform() method and hence the storage of the
 * transformation data.
 */
class AbstractTransformerNode
{
public:

	/// Pointer type for this AbstractTransformerNode
	typedef AbstractTransformerNode* AbstractNodePtr;

protected:
	AbstractTransformerNode(const std::string& id) :
		mID(id), mParent(NULL) {}

	virtual ~AbstractTransformerNode() {}

public:

	/// Returns the id of this node.
	const std::string& getID() const { return mID; }

	/// Returns a list with all children of this node
	const std::list<AbstractNodePtr>& getChildren() const { return mChildren; }

	/// Returns the parent of the node
	const AbstractNodePtr& getParent() const { return mParent; }

	/// Returns the number of descendants of this node (children, grandchildren, etc.)
	uint32 getDescendantCount() const;

	/**
	 * Returns true, if the this node is an ancestor of the given 'other' node.
	 */
	bool isAncestorOf(const AbstractNodePtr other) const;

public:

	/// Returns if the node has data or is empty
	virtual bool isEmpty() const = 0;

	/**
	 * Returns the transform stored in this node at the specified time stamp.
	 * Additionally, the transform can be filtered depending on the specified
	 * filter, e.g. to allow linear interpolation of transforms around the
	 * specified time stamp.
	 * @note This method must be implemented in derived classes !
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const Time& time, Filter&& filter) {
		// #####################################################################
		// If you get a compiler error here, the class that was passed as
		// template parameter TNode does not implement the necessary
		// getTransform() method. Please implement that method in your transform
		// node class!
		// #####################################################################
		static_assert(sizeof(Transform)==0,
			"getTransform() must be implemented in your derived Node class");
		return Transform(); // to prevent compiler warning
	}

protected:
	friend class TransformerBase;

	/// The id of this node.
	std::string                mID;

	/// Pointer to the parent node within the tree
	AbstractNodePtr            mParent;

	/// List with pointers to the children within the transformation tree
	std::list<AbstractNodePtr> mChildren;
};


//////////////////////////////////////////////////////////////////////////////

template <typename T, int D, template <typename T_, int D_> class TTransform>
class Transformer;

/**
 * @ingroup TransformModule
 *
 * Basic reference implementation of an AbstractTransformerNode that stores
 * the transformation data internally in a StampedDataQueue that allows to
 * query the transformation at specific time stamps.
 *
 * Transforms can be added to the internal storage using the setTransform()
 * method.
 *
 * As template parameter the type of the transformation needs to be specified,
 * e.g. RigidTransform3f. This is the type that is internally stored
 * within the storage.
 */
template <typename StorageTransform>
class TransformerNode : public AbstractTransformerNode
{
	typedef AbstractTransformerNode Base;

	/// The type of the internal storage
	typedef StampedDataQueue<StorageTransform> Storage;

protected:
	template <typename T, int D, template <typename T_, int D_> class TTransform_>
	friend class Transformer;

	/**
	 * Creates a new node of the transform tree, that can be
	 * added to the corresponding Transformer using addNode().
	 *
	 * @note The constructors of Node are protected. Use
	 * Transformer::newNode() to create a new node.
	 */
	TransformerNode(const std::string& id) : Base(id) {}

private:
	/// @cond INTERNAL

	// the following two types are used in getTransform(...,Filter&&)
	// they extract the time(as uint64) and the transform out of a
	// Stamped<Transform>
	template <typename Transform>
	struct GetTime
	{
		GetTime(const Time& iT0) : t0(iT0) {}

		typedef int64 result_type;
		result_type operator()(const Stamped<Transform>& p) const {
			return (result_type)(p.timestamp-t0).totalMilliseconds();
		}

		Time t0;
	};

	template <typename Transform>
	struct GetTransform
	{
		typedef const Transform& result_type;
		result_type operator()(const Stamped<Transform>& p) const {
			return p;
		}
	};

	/// @endcond

public:

	/**
	 * Grants access to the internal storage for transformations.
	 * This allows to change the parameters of the storage like the max
	 * storage time, the capacity, etc.
	 */
	Storage& storage() { return mTransformStorage; }

	/**
	 * Inserts the specified transform with the specified time stamp into the
	 * internal storage.
	 */
	void setTransform(const StorageTransform& transform, const Time& time) {
		mTransformStorage.insert(typename Storage::value_type(transform, time));
	}
	/// Implementation of AbstractTransformerNode::isEmpty()
	virtual bool isEmpty() const { return mTransformStorage.empty(); }

	/// Implementation of AbstractTransformerNode::getTransform()
	template<typename Transform, typename Filter>
	Transform getTransform(const Time& time, Filter&& filter)
	{
		// get interval that is required by the filter
		auto its = mTransformStorage.getInterval(time,filter.samples(),
		                                         filter.samplesBefore(),
		                                         filter.samplesAfter());

		const Time t0 = its.first->timestamp;

		// using adapters, iterators and the IteratorRangeContainer, the
		// following code mimics two different containers: one that contains
		// the time as float and one that contains the transforms. In fact
		// there is one container only (mTransformStorage) that contains both
		// within a single container.
		typedef typename Storage::const_iterator const_iterator;
		typedef boost::transform_iterator<GetTime<Transform>, const_iterator> GetTimeIterator;
		GetTimeIterator begin1 = GetTimeIterator(its.first ,GetTime<Transform>(t0));
		GetTimeIterator end1   = GetTimeIterator(its.second,GetTime<Transform>(t0));

		typedef boost::transform_iterator<GetTransform<Transform>, const_iterator> GetTransformIterator;
		GetTransformIterator begin2 = GetTransformIterator(its.first ,GetTransform<Transform>());
		GetTransformIterator end2   = GetTransformIterator(its.second,GetTransform<Transform>());

		typedef mira::IteratorRangeContainer<GetTimeIterator> TimeContainer;
		typedef mira::IteratorRangeContainer<GetTransformIterator> TransformContainer;
		TimeContainer      timeContainer(begin1, end1);
		TransformContainer transformContainer(begin2, end2);

		// call the filter to obtain the filtered transform at the desired time
		return filter.template apply<float, Transform>(timeContainer,
				  transformContainer, (float)(time-t0).totalMilliseconds());
	}

	// specialize for NearestNeighborInterpolator and make a direct call
	// to getData for better performance
	template<typename Transform>
	Transform getTransform(const Time& time, NearestNeighborInterpolator&) {
		return mTransformStorage.getData(time);
	}

protected:

	/// storage that takes all transforms with their timestamps that were added.
	Storage mTransformStorage;
};


//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_TRANSFORMERNODE_H_ */
