/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BufferStream.h
 *    Contains BufferStream a stream for input and output of data to/from
 *    a wrapped Buffer.
 *
 * @author Erik Einhorn
 * @date   2010/11/12
 */

#ifndef _MIRA_BUFFERSTREAM_H__
#define _MIRA_BUFFERSTREAM_H__

#include <iostream>
#include <vector>

#include <platform/Platform.h>
#include <utils/Buffer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// @cond INTERNAL
namespace Private {

/**
 * Implements an iostream buffer for input and output, with separate
 * repositionable reading and writing heads that can be wrapped around a
 * STL vector
 * @ingroup StreamModule
 */
class BufferStreamBuf : public std::streambuf
{
public:

	// STL conform typedefs
	typedef Buffer<char> container_type;
	typedef container_type::value_type char_type;

public:

	/// Creates a new BufferStreamBuf based on the specified std::vector<char> as buffer
	BufferStreamBuf(container_type& buffer);

protected:
	// implements std::streambuf

	virtual int overflow( int c = EOF);
	virtual std::streampos seekoff ( std::streamoff off,
			std::ios_base::seekdir way, std::ios_base::openmode which);
	virtual std::streampos seekpos ( std::streampos pos,
			std::ios_base::openmode which);

private:

	container_type& mBuffer;

	void grow();
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for BufferStream class.
 * @ingroup StreamModule
 */
class BufferStreamBase
{
public:
	typedef BufferStreamBuf::container_type container_type;
public:
	BufferStreamBase(container_type& buffer) : mBuffer(buffer) {}
protected:
	BufferStreamBuf mBuffer;
};

}
/// @endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * A stream class for input and output of data to/from a wrapped Buffer.
 * The usage of this BufferStream is similar to std::stringstream, however,
 * the BufferStream is significantly faster than std::stringstream.
 *
 * The BufferStream can be used as any other STL stream via its implemented
 * << and >> operators.
 *
 * Example:
 * \code
 * // create a buffer for the stream,
 * Buffer<char> buffer;
 *
 * // create the stream that operates on 'buffer'
 * BufferStream stream(buffer);
 *
 * // write something into the stream
 * stream << "This is a test: " << 1234 << ", " << 12.34f;
 *
 * // read from stream
 * std::string s;
 * stream >> s;
 * \endcode
 *
 * @ingroup StreamModule
 */
class BufferStream : public Private::BufferStreamBase, public std::iostream
{
public:

	// STL conform typedefs
	typedef Buffer<char> container_type;
	typedef container_type::value_type char_type;

public:
	BufferStream(Buffer<char>& buffer) :
		BufferStreamBase(buffer), std::iostream(&mBuffer) {}

};


///////////////////////////////////////////////////////////////////////////////
}

#endif
