/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TypeId.h
 *    Provides method for generating a unique id for any type.
 *
 * @author Erik Einhorn
 * @date   2010/07/01
 */

#ifndef _MIRA_TYPEID_H_
#define _MIRA_TYPEID_H_

#include <typeinfo>
#ifndef Q_MOC_RUN
#include <boost/type_traits/remove_cv.hpp>
#endif
#include <platform/Platform.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * The type of the integral TypeId, that can be retrieved by typeId<T>()
 * @ingroup ClassFactoryModule
 */
typedef int TypeId;

///@cond INTERNAL
namespace Private {

struct TypeIdHelperBase
{
	static TypeId nextId(const std::type_info& type);
};

template <typename T>
struct TypeIdHelper : public TypeIdHelperBase
{
	static TypeId id() {
		static TypeId id = nextId(typeid(T));
		return id;
	}
};

}
///@endcond

/**
 * Generates unique IDs for different types.
 * However, the IDs may change with any start of the application. Therefore,
 * they only can be used for within one process, e.g. for checking if
 * two types are identical, storing type information, etc.
 * @ingroup ClassFactoryModule
 */
template<typename T>
TypeId typeId()
{
	// make type id for unqualified type (remove_cv is used to remove const, etc)
	return Private::TypeIdHelper< typename boost::remove_cv<T>::type >::id();
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
