/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file polygon_example.C
 *    An example how to use rasterization.
 *
 * @author Jens Kessler
 * @date   2010/10/1
 */

#include <iostream>
#include <opencv2/highgui/highgui.hpp>

#include <image/Img.h>

#include <geometry/Geometry.h>
#include <geometry/Bresenham.h>
#include <geometry/RasterTriangle.h>
#include <geometry/RasterRect.h>

using namespace mira;
using namespace boost::geometry;

///////////////////////////////////////////////////////////////////////////////

/// A vistor class for rectangle. Each visited pixel will be painted white.
class RectVisitor
{
public:
	RectVisitor(Img8U1& img) : myImg(img) {}

	// the rasterization callback for runslice algorithm
	void operator()(int x, int y, int length)
	{
		uint8* tStart=myImg[y];
		tStart += x;
		for(int i=0; i<length; i++)
		{
			// paint it white
			*tStart = 220;
			tStart++;
		}
	}
private:
	Img8U1& myImg;
};

///////////////////////////////////////////////////////////////////////////////

/// A visitor class for a triangle. Each visited pixel will be painted white.
class TriangleVisitor
{
public:
	// The constructor.
	TriangleVisitor(Img8U1& img) : myImg(img) {}

	// The rasterization callback for runslice algorithm.
	void operator()(int xl, int xr, int y)
	{
		uint8* tStart = myImg[y];
		tStart += xl;
		for(int i = xl; i < xr; i++)
		{
			// paint it white
			*tStart = 220;
			tStart++;
		}
	}

private:
	Img8U1& myImg;
};

///////////////////////////////////////////////////////////////////////////////

int main()
{
	// Create line
	Point2i A(5, 0);
	Point2i B(10, 100);
	Line2i  line(A, B); // start and end point

	// Create a rectangle (lower left point, width, height)
	Rect2i rect(Point2i(30, 10), 20, 10);

	// Create a triangle
	Polygon2i triangle = createTriangle(Point2i(50, 100),
	                                    Point2i(70, 120),
	                                    Point2i(40, 110));

	// create the image and assign it to the visitors
	Img8U1 myImage(300, 300);
	//initialize image with black
	myImage = 0;
	RectVisitor     visitor1(myImage);
	TriangleVisitor visitor2(myImage);

	// raster an rect
	rasterRect(rect, visitor1);

	// raster a triangle
	rasterTriangle(triangle, visitor2);

	// raster a line
	bresenhamRunSlice(line.first, line.second, visitor1);

	// save the picture
	cv::imwrite("Test.png", (cv::Mat&)myImage);

	return 0;
}
